<?php

namespace Epaka\Service;

class LoggerService
{
    private const LOG_DIR = __DIR__ . '/../../logs';
    private const MAX_LOG_DAYS = 5;

    public static function logError(string $message, int $code, string $file, string $context): void
    {
        if (!file_exists(self::LOG_DIR)) {
            mkdir(self::LOG_DIR, 0777, true);
        }

        $date = date('Y-m-d');
        $logFile = self::LOG_DIR . "/$date.log";

        $logMessage = sprintf(
            "[%s] Error: %s | Code: %d | File: %s | Context: %s%s%s%s",
            date('Y-m-d H:i:s'),
            $message,
            $code,
            $file,
            $context,
            PHP_EOL,
            str_repeat("-", 80),
            PHP_EOL
        );

        file_put_contents($logFile, $logMessage, FILE_APPEND);
        self::cleanupOldLogs();
    }

    private static function cleanupOldLogs(): void
    {
        $files = glob(self::LOG_DIR . '/*.log');
        if (count($files) > self::MAX_LOG_DAYS) {
            array_multisort(array_map('filemtime', $files), SORT_ASC, $files);
            foreach (array_slice($files, 0, count($files) - self::MAX_LOG_DAYS) as $file) {
                unlink($file);
            }
        }
    }

    public static function normalizeErrorToFlashMessage(?array $errorsRaw, string $defaultErrorMessage ): array
    {
        return [
            'message' => self::normalizeErrorMessage($errorsRaw, $defaultErrorMessage)
        ];
    }

    public static function normalizeErrorToCollection(int $currentRecordId, ?array $errorsRaw, string $defaultErrorMessage ): array
    {
        return [
            'id' => $currentRecordId,
            'message' => self::normalizeErrorMessage($errorsRaw, $defaultErrorMessage)
        ];
    }

    private static function normalizeErrorMessage(?array $errorsRaw, string $defaultErrorMessage): string
    {
        $errorsToMessage = '';
        if (json_last_error() === JSON_ERROR_NONE && is_array($errorsRaw)) {
            foreach ($errorsRaw as $error) {
                switch (true) {
                    case isset($error['field']) && isset($error['field']): {
                        $errorsToMessage .= "{$error['field']}: {$error['message']}<br>";
                        break;
                    }
                    case isset($error['field']): {
                        $errorsToMessage .= "{$error['field']}<br>";
                    }
                    case isset($error['message']): {
                        $errorsToMessage .= "{$error['message']}<br>";
                    }
                }
            }
        }
        $errorsToMessage = mb_convert_encoding($errorsToMessage, 'UTF-8', 'auto');

        return $errorsToMessage !== '' ? $errorsToMessage : $defaultErrorMessage;
    }

    public static function normalizeCollectionFlashMessages(array $errors, string $errorFlashMessage, string $successFlashMessage = 'Akcja zakończyła się sukcesem.', string $messageType = 'warning'): array
    {
        if (empty($errors)) {
            return [
                'message' => $successFlashMessage,
                'type' => 'success'
            ];
        }
        $errorDetails = [];
        foreach ($errors as $error) {
            if (isset($error['id']) && $error['id'] !== '' && $error['id'] !== null) {
                $errorDetails[] = "<span style='color: red;'><b>Rekord ID {$error['id']}:</b></span><br>{$error['message']}";
            } elseif (isset($error['message'])) {
                $errorDetails[] = $error['message'];
            }
        }
        $errorRowsDetailsInString = implode('<br>', $errorDetails);
        $errorFlashMessage = '<b>' . $errorFlashMessage . ':</b><br><br>' . $errorRowsDetailsInString;
        return [
            'message' => $errorFlashMessage,
            'type' => $messageType
        ];
    }
}
