<?php

namespace Epaka\Service;

use Exception;

class CryptoService
{
    private const CIPHER = 'AES-256-CBC';
    
    /**
     * Encrypts a string using the specified key
     *
     * @param string $plainText Text to encrypt
     * @param string $key Encryption key
     * @return string Encrypted string in base64
     * @throws Exception
     */
    public static function encrypt(string $plainText, string $key): string
    {
        try {
            // Generate IV (initialization vector)
            $ivLength = openssl_cipher_iv_length(self::CIPHER);
            $iv = openssl_random_pseudo_bytes($ivLength);
            
            // Encrypt the data
            $encrypted = openssl_encrypt(
                $plainText,
                self::CIPHER,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );
            
            if ($encrypted === false) {
                throw new Exception('Encryption failed: ' . openssl_error_string());
            }
            
            // Combine IV and encrypted data and encode in base64
            return base64_encode($iv . $encrypted);
            
        } catch (Exception $e) {
            throw new Exception('Encryption error: ' . $e->getMessage());
        }
    }
    
    /**
     * Decrypts a string using the specified key
     *
     * @param string $encryptedText Encrypted string in base64
     * @param string $key Encryption key
     * @return string Decrypted string
     * @throws Exception
     */
    public static function decrypt(string $encryptedText, string $key): string
    {
        try {
            // Decode base64
            $decoded = base64_decode($encryptedText, true);
            
            if ($decoded === false) {
                throw new Exception('Invalid base64 encoding');
            }
            
            // Get IV length
            $ivLength = openssl_cipher_iv_length(self::CIPHER);
            
            if (strlen($decoded) < $ivLength) {
                throw new Exception('Invalid encrypted data');
            }
            
            // Extract IV and encrypted data
            $iv = substr($decoded, 0, $ivLength);
            $encrypted = substr($decoded, $ivLength);
            
            // Decrypt
            $decrypted = openssl_decrypt(
                $encrypted,
                self::CIPHER,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );
            
            if ($decrypted === false) {
                throw new Exception('Decryption failed: ' . openssl_error_string());
            }
            
            return $decrypted;
            
        } catch (Exception $e) {
            throw new Exception('Decryption error: ' . $e->getMessage());
        }
    }
    
    /**
     * Generates a random encryption key
     *
     * @param int $length Key length
     * @return string Generated key
     */
    public static function generateKey(int $length = 32): string
    {
        return bin2hex(random_bytes($length));
    }

    /**
     * Encrypts a string using a deterministic approach (always produces the same result for the same input)
     *
     * @param string $plainText Text to encrypt
     * @param string $key Encryption key
     * @return string Encrypted string in base64
     * @throws Exception
     */
    public static function encryptStatic(string $plainText, string $key): string
    {
        try {
            $ivLength = openssl_cipher_iv_length(self::CIPHER);
            $hashData = hash('sha256', $plainText . $key, true);
            $iv = substr($hashData, 0, $ivLength);

            $encrypted = openssl_encrypt(
                $plainText,
                self::CIPHER,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );

            if ($encrypted === false) {
                throw new Exception('Static encryption failed: ' . openssl_error_string());
            }

            return bin2hex($iv . $encrypted);
        } catch (Exception $e) {
            throw new Exception('Static encryption error: ' . $e->getMessage());
        }
    }

    /**
     * Validates that an encrypted static token matches the expected plain text
     */
    public static function decryptStatic(string $encryptedText, string $key): string
    {
        try {
            $binary = hex2bin($encryptedText);

            if ($binary === false) {
                throw new Exception('Invalid hex encoding');
            }
            $ivLength = openssl_cipher_iv_length(self::CIPHER);
            $iv = substr($binary, 0, $ivLength);
            $encrypted = substr($binary, $ivLength);

            $decrypted = openssl_decrypt(
                $encrypted,
                self::CIPHER,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );

            if ($decrypted === false) {
                throw new Exception('Decryption failed: ' . openssl_error_string());
            }

            return $decrypted;
        } catch (Exception $e) {
            throw new Exception('Static decryption error: ' . $e->getMessage());
        }
    }
}