<?php

namespace Epaka\Service;

use Exception;
use Module;

class ApiService
{
//    public const DEFAULT_URL = 'https://api.devjcd.epaka.pl';
//    public const DEFAULT_MAP_URL = 'https://devjcd.epaka.pl';
    public const DEFAULT_URL = 'https://api.epaka.pl';
    public const DEFAULT_MAP_URL = 'https://www.epaka.pl';
    private const OAUTH_URL = '/oauth/token';
    private const COURIERS_URL = '/v1/order/prices';
    private const ADDITIONAL_FIELDS_URL = '/v1/order/services';
    private const PACKAGES_TYPE_URL = '/v1/order/package-type';
    private const INSURANCE_RECALCULATE_URL = '/v1/order/services/insurance/price';
    private const COD_RECALCULATE_URL = '/v1/order/services/cod/prices';
    private const PICKUP_DAY_URL = '/v1/order/pickup-date';
    private const PICKUP_TIME_URL = '/v1/order/pickup-hours';
    private const ORDER_URL = '/v1/order';
    private const ORDER_CHECK_DATA_URL = '/v1/order/check-data';
    private const COR_RETURN_TYPE_URL = '/v1/order/services/cod';
    private const CANCEL_ORDER_URL = '/v1/user/orders/:orderId/cancel';
    private const DETAIL_ORDER_URL = '/v1/user/orders';
    private const BALANCE = '/v1/user/balance';
    private const USER_INFO = '/v1/user';

    private static function getModuleVersion(): string
    {
        $module = Module::getInstanceByName('epaka');
        return $module->version;
    }
    
    /**
     * Base method for executing OAuth requests
     */
    private static function makeOAuthRequest(array $data, string $clientId, string $clientSecret): array
    {
        $ch = curl_init();
        $options = [
            CURLOPT_URL => self::DEFAULT_URL . self::OAUTH_URL,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Basic ' . base64_encode($clientId . ':' . $clientSecret),
                'Content-Type: application/x-www-form-urlencoded',
                'app-os: PrestaShop',
                'app-version: ' . self::getModuleVersion()
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_ENCODING => '',
        ];
        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new Exception('Curl error: ' . $error);
        }

        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        $decodedResponse = json_decode($response, true);


        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON response: ' . json_last_error_msg());
        }

        switch ($httpCode) {
            case 200:
                return $decodedResponse;
            case 400:
                throw new Exception('Bad Request');
            case 403:
                throw new Exception('Forbidden');
            case 503:
                throw new Exception('Service Unavailable');
            default:
                throw new Exception('Unknown error');
        }
    }
    public static function fetchToken(string $email, string $password, string $clientId, string $clientSecret): array
    {
        $data = [
            'grant_type' => 'password',
            'username' => $email,
            'password' => $password,
        ];
        
        return self::makeOAuthRequest($data, $clientId, $clientSecret);
    }
    public static function refreshToken(string $clientId, string $clientSecret, string $refreshToken): array
    {
        $data = [
            'grant_type' => 'refresh_token',
            'refresh_token' => $refreshToken
        ];
        
        return self::makeOAuthRequest($data, $clientId, $clientSecret);
    }

    /**
     * Base method for executing API requests with a token
     * @throws Exception
     */
    private static function makeApiRequest(string $method, string $endpoint, array $data = [], string $accessToken = null, bool $isForTest = false): array
    {
        $ch = curl_init();
        $headers = [
            'Content-Type: application/json',
            'app-os: PrestaShop',
            'app-version' => self::getModuleVersion()
        ];
        if ($accessToken) {
            $headers[] = 'Authorization: Bearer ' . $accessToken;
        }
        $options = [
            CURLOPT_URL => self::DEFAULT_URL . $endpoint,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_ENCODING => '',
        ];
        if ($method === 'POST') {
            $options[CURLOPT_POST] = true;
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        } elseif ($method === 'PUT') {
            $options[CURLOPT_CUSTOMREQUEST] = 'PUT';
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        } elseif ($method === 'PATCH') {
            $options[CURLOPT_CUSTOMREQUEST] = 'PATCH';
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        } elseif ($method === 'DELETE') {
            $options[CURLOPT_CUSTOMREQUEST] = 'DELETE';
        } elseif (!empty($data)) {
            $options[CURLOPT_URL] .= '?' . http_build_query($data);
        }
        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new Exception('Curl error: ' . $error);
        }
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        $decodedResponse = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON response: ' . json_last_error_msg());
        }
        if ($httpCode >= 200 && $httpCode < 300) {
            return $decodedResponse;
        }
        $errorMessage = 'API error';
        if (isset($decodedResponse['errors']) && is_array($decodedResponse['errors'])) {
            $errorMessage = json_encode($decodedResponse['errors'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        } elseif (isset($decodedResponse['message'])) {
            $errorMessage = $decodedResponse['message'];
        }
        throw new Exception($errorMessage, $httpCode);
    }

    /**
     * @throws Exception
     */
    public static function getCouriers(string $accessToken, array $packages, string $shippingType, string $senderCountry, string $receiverCountry, int $courierId = null): array
    {
        $packagesFromApi = self::getPackagesType($accessToken, $shippingType);
        $packages = self::preparePackagesForCourierRequest($packages, $packagesFromApi);
        $body = array_filter([
            'packages' => $packages,
            'shippingType' => $shippingType,
            'courierId' => $courierId,
            'senderCountry' => $senderCountry,
            'receiverCountry' => $receiverCountry,
        ]);
        try {
            return self::makeApiRequest('POST', self::COURIERS_URL, $body, $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    private static function preparePackagesForCourierRequest(array $frontendData, array $packagesFromApi): array
    {
        return array_map(function($package) use ($packagesFromApi) {
            $isFindResultInApi = false;
            foreach ($packagesFromApi as $packFromApi) {
                foreach ($packFromApi as $apiPackage) {
                    if(trim(strtolower($package['packageType'])) === trim(strtolower($apiPackage['name']))) {
                        $package['type'] = (int) $apiPackage['id'];
                        $isFindResultInApi = true;
                        break;
                    }
                }
            }

            // The Epaka API did not return a response for 0 (standardowe) and 1 (niestandardowe),
            // but they can actually be referenced using these IDs.
            // This is only an issue for "envelop".
            if(!$isFindResultInApi) {
                if(trim(strtolower($package['packageType'])) === ParcelTypeHardcodedService::getPackageType(0)['name']) {
                    $package['type'] = ParcelTypeHardcodedService::getPackageType(0)['id'];
                } elseif (trim(strtolower($package['packageType'])) === ParcelTypeHardcodedService::getPackageType(1)['name']) {
                    $package['type'] = ParcelTypeHardcodedService::getPackageType(1)['id'];
                }
            }
            unset($package['packageType']);
            return $package;
        }, $frontendData);
    }

    /**
     * @throws Exception
     */
    public static function getPackagesType(string $accessToken, string $shippingType): array
    {
        $filters = [
            'shippingType' => $shippingType
        ];
        try {
            return self::makeApiRequest('GET', self::PACKAGES_TYPE_URL, $filters, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }

     /**
     * @throws Exception
     */
    public static function getPickUpDays(string $accessToken, int $courierId): array
    {
        $filters = [
            'couriers' => $courierId
        ];
        try {
            return self::makeApiRequest('GET', self::PICKUP_DAY_URL, $filters, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function getPickUpTime(string $accessToken, int $courierId, string $data, string $postalCode, string $senderCountryIsoCode, ?string $dhlType = null, ?string $upsServiceType = null, ?int $upsPackagesNumber = null, ?float $upsWeight = null): array
    {
        $filters = array_filter([
            'courierId' => $courierId,
            'date' => $data,
            'packagesNumber' => $upsPackagesNumber,
            'postCode' => $postalCode,
            'senderCountry' => $senderCountryIsoCode,
            'serviceType' => $upsServiceType,
            'type' => $dhlType,
            'weight' => $upsWeight,
        ]);
        try {
            return self::makeApiRequest('GET', self::PICKUP_TIME_URL, $filters, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function createOrder(
        string $accessToken,
        int $courierId,
        string $productsNames,
        string $paymentType,
        string $wcOrderContentName,
        string $pickupDate,
        ?string $pickupTimeFrom,
        ?string $pickupTimeTo,
        array $packages,
        array $receiver,
        array $sender,
        string $shippingType,
        ?string $carrierUserComment,
        ?float $insuranceAmount,
        ?float $codeAmount = null,
        ?array $additionalOptionsNameCollection = null,
        ?string $codReturnType = null
    ): array
    {
        try {
            $returnTypes = self::fetchReturnTypes($accessToken, $courierId)['codTypes'];
            $codType = $codeAmount ? self::getCurrentCodType($returnTypes) : null;
            $data = static::array_filter_recursive([
                'content' => $wcOrderContentName,
                'courierId' => $courierId,
                'packages' => $packages,
                'paymentData' => [
                    'paymentType' => $paymentType
                ],
                'pickupDate' => $pickupDate,
                'pickupTime' => [
                    'from' => $pickupTimeFrom,
                    'to' => $pickupTimeTo
                ],
                'receiver' => $receiver,
                'sender' => $sender,
                'shippingType' => $shippingType,
                'comments' => $carrierUserComment,
                'services' => [
                    'cod' => (bool)$codeAmount,
                    'codAmount' => $codeAmount,
                    'codReturnPointId' => null,
                    'codReturnType' => $codReturnType,
                    'codType' => $codType,
                    'additionalServices' => $additionalOptionsNameCollection,
                    'declaredValue' => $insuranceAmount,
                    'insurance' => (bool)$insuranceAmount
                ]
            ]);
            return self::makeApiRequest('POST', self::ORDER_URL, $data, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }
    private static function array_filter_recursive($array): array
    {
        foreach ($array as $key => &$value) {
            if (is_array($value)) {
                $value = static::array_filter_recursive($value);
            }
        }
        return array_filter($array, function ($value) {
            return !is_null($value) && $value !== '' && $value !== false;
        });
    }

    /**
     * @throws Exception
     */
    public static function cancelOrder(string $accessToken, int $orderId)
    {
        try {
            $endpoint = str_replace(':orderId', $orderId, self::CANCEL_ORDER_URL);
            return self::makeApiRequest('PATCH', $endpoint, [], $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function fetchOrderDetails(string $accessToken, int $orderId): array
    {
        try {
            return self::makeApiRequest('GET', self::DETAIL_ORDER_URL . '/'. $orderId, [], $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    private static function fetchReturnTypes(string $accessToken, int $courierId): array
    {
        $filters = [
            'courierId' => $courierId
        ];
        try {
            return self::makeApiRequest('GET', self::COR_RETURN_TYPE_URL, $filters, $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }
    private static function getCurrentCodType(array $codTypes): string
    {
        foreach ($codTypes as $codType) {
            if($codType['apiName'] === 'cod.saldo') return 'cod.saldo';
        }
        return current($codTypes)['apiName'];
    }

    /**
     * @throws Exception
     */
    public static function getServices(string $accessToken, ?int $courierId, string $senderCountry, string $receiverCountry, string $shippingType, ?string $senderPostCode = null, string $receiverPostCode = null, ?array $existingAdditionalFieldsData = null): array
    {
        $filters = array_filter([
            'courierId' => $courierId,
            'senderCountry' => $senderCountry,
            'receiverCountry' => $receiverCountry,
            'shippingType' => $shippingType,
            'senderPostCode' => $senderPostCode,
            'receiverPostCode' => $receiverPostCode
        ]);
        try {
            $courierAdditionalFields = self::makeApiRequest('GET', self::ADDITIONAL_FIELDS_URL, $filters, $accessToken);
            $courierAdditionalFields['services'] = self::normalizeCourierAdditionalFieldsApiData($courierAdditionalFields, $existingAdditionalFieldsData);
            return $courierAdditionalFields;
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }
    private static function normalizeCourierAdditionalFieldsApiData(array $additionalFields, ?array $existingAdditionalFieldsData = null): array
    {
        $availableServices = array_filter(
            $additionalFields['services'],
            function ($service) {
                return $service['available'] === true;
            }
        );
        return array_map(
            function ($service) use ($existingAdditionalFieldsData) {
                $service['label'] = $service['name'];
                $service['name'] = $service['apiName'];
                $service['maxValue'] = $service['insuranceMaxValue'] ?? $service['codMaxValue'] ?? null;
                unset($service['insuranceMaxValue'], $service['codMaxValue']);
                $service['extra_field'] = in_array($service['apiName'], ['cod', 'insurance'])
                    ? ['name' => $service['apiName'] . '_amount', 'label' => $service['label']]
                    : null;
                $service['fieldState'] = ['isActive' => false];
                if ($existingAdditionalFieldsData) {
                    $matchedField = array_filter(
                        $existingAdditionalFieldsData,
                        function ($fieldName) use ($service) {
                            return trim(strtolower($service['apiName'])) === trim(strtolower($fieldName));
                        },
                        ARRAY_FILTER_USE_KEY
                    );
                    if ($matchedField) {
                        $fieldValue = reset($matchedField);
                        $service['fieldState']['isActive'] = true;
                        $service['fieldState']['amount'] = is_array($fieldValue) && isset($fieldValue['amount'])
                            ? $fieldValue['amount']
                            : null;
                        $service['fieldState']['isActive'] = is_array($fieldValue)
                            ? (isset($fieldValue['value']) && $fieldValue['value'] === '1') || isset($fieldValue['amount'])
                            : $fieldValue === '1';
                    }
                }
                return $service;
            },
            $availableServices
        );
    }

    /**
     * @throws Exception
     */
    public static function getInsurancePrice(string $accessToken, int $courierId, string $senderCountry, string $receiverCountry, string $maxWeight, string $declaredValue): array
    {
        $filters = [
            'courierId' => $courierId,
            'senderCountry' => $senderCountry,
            'receiverCountry' => $receiverCountry,
            'maxWeight' => $maxWeight,
            'declaredValue' => $declaredValue
        ];
        try {
            return self::makeApiRequest('GET', self::INSURANCE_RECALCULATE_URL, $filters, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function getCodPrice(string $accessToken, int $courierId, float $amount): array
    {
        $filters = [
            'courierId' => $courierId,
            'codAmount' => $amount
        ];
        try {
            return self::makeApiRequest('GET', self::COD_RECALCULATE_URL, $filters, $accessToken);
        } catch (Exception $e){
            Throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function fetchAccountBalance(string $accessToken): float
    {
        try {
            $response =  self::makeApiRequest('GET', self::BALANCE, [], $accessToken);
            return $response['balance'];
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function fetchUserInfo(string $accessToken): array
    {
        try {
            return self::makeApiRequest('GET', self::USER_INFO, [], $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function fetchOrderDocument(string $accessToken, int $orderId, string $type): array
    {
        $url = self::DETAIL_ORDER_URL . '/' . $orderId . '/' . $type;
        try {
            return self::makeApiRequest('GET', $url, [], $accessToken);
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }
}