<?php

namespace Epaka\Service;

use Epaka\Entity\EpakaSender;
use Epaka\Mapper\ShippingMapper;
use PrestaShop\PrestaShop\Core\Foundation\IoC\Exception;

final class ApiOrderService
{
    private static $maxHeight = 0;
    private static $maxWidth = 0;
    private static $maxDepth = 0;
    private static $totalWeight = 0;
    private static $productsNames = [];

    /**
     * @throws Exception
     * @return array|false Returns an array on success or false on failure.
     */
    public static function create(
        string $accessToken,
        int $packageType,
        int $parcelType,
        string $paymentType,
        string $wcOrderContentName,
        array $products,
        int $epakaCourierId,
        float $carrierMaxHeight,
        float $carrierMaxDepth,
        float $carrierMaxWidth,
        float $cartTotalWeight,
        string $deliveryCity,
        string $deliveryCountryIsoCode,
        string $customerEmail,
        string $deliveryHouseNumber,
        string $deliveryFirstName,
        string $deliveryLastName,
        string $deliveryPhone,
        string $deliveryPostCode,
        string $deliveryStreet,
        ?string $deliveryVatNumber,
        ?string $deliveryCompany,
        EpakaSender $epakaSender,
        ?string $carrierUserComment = null,
        ?float $insuranceAmount = null,
        ?float $codeAmount = null,
        ?array $additionalOptionsNameCollection = null,
        ?string $codReturnType = null,
        ?string $trackingNumber = null,
        ?string $dhlType = null,
        ?string $upsServiceType = null,
        ?int $upsPackagesNumber = null,
        ?float $upsWeight = null
    )
    {
        foreach ($products as $product) {
            self::$productsNames[] = $product['name'];
            self::$maxHeight = max(self::$maxHeight, (float)$product['height']);
            self::$maxWidth = max(self::$maxWidth, (float)$product['width']);
            self::$maxDepth = max(self::$maxDepth, (float)$product['depth']);
            self::$totalWeight += (float)$product['weight'] * (int)$product['cart_quantity'];
        }
        $packageContent = implode(', ', self::$productsNames);
        try {
            $packages = [
                [
                    'height' => round(self::$maxHeight > 0 ? self::$maxHeight : $carrierMaxHeight, 2),
                    'length' => round(self::$maxDepth > 0 ? self::$maxDepth : $carrierMaxDepth, 2),
                    'width' => round(self::$maxWidth > 0 ? self::$maxWidth : $carrierMaxWidth, 2),
                    'weight' => round(self::$totalWeight > 0 ? self::$totalWeight : $cartTotalWeight, 2),
                    'type' => self::fetchParcelType($accessToken, $packageType, $parcelType)
                ]
            ];
            $pickupDate = self::fetchPickUpDay($accessToken, $epakaCourierId);

            [$pickupTimeFrom, $pickupTimeTo] = self::fetchPickUpTime(
                $accessToken,
                $epakaCourierId,
                $pickupDate,
                $deliveryPostCode,
                $deliveryCountryIsoCode,
                $dhlType,
                $upsServiceType,
                $upsPackagesNumber,
                $upsWeight
            );
            $receiver = array_filter([
                'city' => $deliveryCity,
                'country' => $deliveryCountryIsoCode,
                'email' => $customerEmail,
                'houseNumber' => $deliveryHouseNumber,
                'lastName' => $deliveryLastName,
                'name' => $deliveryFirstName,
                'phone' => $deliveryPhone,
                'postCode' => $deliveryPostCode,
                'street' => $deliveryStreet,
                'company' => $deliveryCompany,
                'nip' => $deliveryVatNumber,
                'pointId' => $trackingNumber,
                'pointDescription' => $trackingNumber
            ], function ($value) {
                return !is_null($value) && $value !== '' && $value !== false;
            });
            $sender = [
                'city' => $epakaSender->getCity(),
                'country' => $epakaSender->getCountryIsoCode(),
                'company' => $epakaSender->getCompanyName(),
                'email' => $epakaSender->getEmail(),
                'houseNumber' => $epakaSender->getHouseNumber(),
                'lastName' => $epakaSender->getLastName(),
                'name' => $epakaSender->getFirstName(),
                'phone' => $epakaSender->getPhone(),
                'postCode' => $epakaSender->getPostalCode(),
                'street' => $epakaSender->getStreet(),
                'pointId' => $trackingNumber,
                'pointDescription' => $trackingNumber
            ];
            $shippingType = ShippingMapper::convertShippingTypeToString($packageType);

            return self::createOrder(
                $accessToken,
                $epakaCourierId,
                $packageContent,
                $paymentType,
                $wcOrderContentName,
                $pickupDate,
                $pickupTimeFrom,
                $pickupTimeTo,
                $packages,
                $receiver,
                $sender,
                $shippingType,
                $carrierUserComment,
                $insuranceAmount,
                $codeAmount,
                $additionalOptionsNameCollection,
                $codReturnType
            );
        } catch (\Exception $e) {
            throw new Exception($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws \Exception
     */
    private static function fetchParcelType(string $accessToken, int $packageType, int $parcelType): int
    {
        try {
            $epakaApiParcelTypes = ApiService::getPackagesType($accessToken, ShippingMapper::convertShippingTypeToString($packageType));
            foreach ($epakaApiParcelTypes['types'] as $epakaApiParcelType) {
                if(trim(strtolower($epakaApiParcelType['name'])) === trim(strtolower(ShippingMapper::convertParcelTypeToString($parcelType)))) {
                    return $epakaApiParcelType['id'];
                }
            }
            // The Epaka API did not return a response for 0 (standardowe) and 1 (niestandardowe),
            // but they can actually be referenced using these IDs.
            // This is only an issue for "envelop".
            if(trim(strtolower(ShippingMapper::convertParcelTypeToString($parcelType))) === ParcelTypeHardcodedService::getPackageType(0)['name']) {
                return ParcelTypeHardcodedService::getPackageType(0)['id'];
            } elseif (trim(strtolower(ShippingMapper::convertParcelTypeToString($parcelType))) === ParcelTypeHardcodedService::getPackageType(1)['name']) {
                return ParcelTypeHardcodedService::getPackageType(1)['id'];
            }
        } catch (\Exception $e) {
            throw new \Exception(
                $e->getMessage(),
                $e->getCode()
            );
        }
    }

    /**
     * @throws \Exception
     */
    private static function fetchPickUpDay(string $accessToken, int $epakaCourierId): string
    {
        try {
            $response = ApiService::getPickUpDays($accessToken, $epakaCourierId);
            if (!empty($response['couriers']) && is_array($response['couriers'])) {
                $firstCourier = reset($response['couriers']);
                return $firstCourier['startDate'];
            } else {
                throw new \Exception('Pickup date not found (Create epaka order) (Get pickup day) (Pickup date)"', 502);
            }
        } catch (\Exception $e) {
            throw new \Exception(
                $e->getMessage(),
                $e->getCode()
            );
        }
    }

    /**
     * @throws Exception
     * @throws \Exception
     */
    private static function fetchPickUpTime(string $accessToken, int $epakaCourierId, string $pickUpDay, string $postCode, string $countryIsoCode, ?string $dhlType = null, ?string $upsServiceType = null, ?int $upsPackagesNumber = null, ?float $upsWeight = null): ?array
    {
        try {
            $pickupTimes = ApiService::getPickUpTime(
                $accessToken,
                $epakaCourierId,
                $pickUpDay,
                $postCode,
                $countryIsoCode,
                $dhlType,
                $upsServiceType,
                $upsPackagesNumber,
                $upsWeight
            );

            if(empty($pickupTimes)) return null;
            if (!empty($pickupTimes['timeSlots']) && is_array($pickupTimes['timeSlots'])) {
                $firstSlot = current($pickupTimes['timeSlots']);
                if ($firstSlot !== false && isset($firstSlot['timeFrom'], $firstSlot['timeTo'])) {
                    return [$firstSlot['timeFrom'], $firstSlot['timeTo']];
                } else {
                    throw new Exception('"timeFrom" or "timeTo" not found (Create epaka order) (Get pickup time) (timeFrom or timeTo)', 502);
                }
            }
        } catch (\Exception $e) {

            throw new \Exception(
                $e->getMessage(),
                $e->getCode()
            );
        }
    }


    /**
     * @throws \Exception
     */
    private static function createOrder(
        string $accessToken,
        int $courierId,
        string $productsNames,
        string $paymentType,
        string $wcOrderContentName,
        string $pickupDate,
        ?string $pickupTimeFrom,
        ?string $pickupTimeTo,
        array $packages,
        array $receiver,
        array $sender,
        string $shippingType,
        ?string $carrierUserComment = null,
        ?float $insuranceAmount = null,
        ?float $codeAmount = null,
        ?array $additionalOptionsNameCollection = null,
        ?string $codReturnType = null
    ): array
    {
        try {
            return  ApiService::createOrder(
                $accessToken,
                $courierId,
                $productsNames,
                $paymentType,
                $wcOrderContentName,
                $pickupDate,
                $pickupTimeFrom,
                $pickupTimeTo,
                $packages,
                $receiver,
                $sender,
                $shippingType,
                $carrierUserComment,
                $insuranceAmount,
                $codeAmount,
                $additionalOptionsNameCollection,
                $codReturnType
            );
        } catch (\Exception $e) {
            throw new \Exception(
                $e->getMessage(),
                $e->getCode()
            );
        }
    }
}