<?php

namespace Epaka\Controller;

use Carrier;
use Configuration;
use Context;
use Country;
use DateTime;
use Db;
use Epaka\Entity\EpakaCarrier;
use Epaka\Entity\EpakaSender;
use Epaka\Entity\EpakaSettings;
use Epaka\Form\AdditionalCourierSettingsFormType;
use Epaka\Form\CourierFormType;
use Epaka\Form\DefaultSenderFormType;
use Epaka\Form\PriceToFrontendFormType;
use Epaka\Form\ShippingFormType;
use Epaka\Mapper\ShippingMapper;
use Epaka\Service\ApiService;
use Epaka\Service\LoggerService;
use Epaka\Traits\CountryTrait;
use Exception;
use Group;
use Language;
use PrestaShop\PrestaShop\Adapter\Country\CountryDataProvider;
use PrestaShop\PrestaShop\Adapter\Validate;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use RangeWeight;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tools;
use Zone;

class ShippingController extends FrameworkBundleAdminController
{
    use CountryTrait;
    private $context;

    public function __construct()
    {
        parent::__construct();
        $this->context = Context::getContext();
    }

    public function setCountryDataProvider(CountryDataProvider $countryDataProvider): void
    {
        $this->countryDataProvider = $countryDataProvider;
    }

    public function listAction(Request $request)
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $em = $this->getDoctrine()->getManager();
        $carriers = $em->getRepository(EpakaCarrier::class)->findBy(['isDeleted' => false], ['id' => 'DESC']);
        $carriers = array_map(function ($carrier) {
            $psCarrier = new Carrier($carrier->getPsCarrierId());
            $prices = $this->getCarrierZonesData($psCarrier->id);
            $carrier->name = $psCarrier ? ($psCarrier->name ?? 'Brak danych') : 'Brak danych';
            $carrier->isActive = $psCarrier && isset($psCarrier->active) ? $psCarrier->active : 'Brak danych';
            $carrier->price = $prices ? round($prices[0]['price'], 2) : '-';
            return $carrier;
        }, $carriers);
        return $this->render('@Modules/epaka/views/templates/admin/shipping/list.html.twig', [
            'carriers' => $carriers,
        ]);
    }

    public function fakeListAction()
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        return $this->render('@Modules/epaka/views/templates/admin/shipping/fakeList.html.twig');
    }

    /**
     * @throws Exception
     */
    public function addAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        [$couriersUrl, $additionalCourierSettingsUrl, $priceToFrontendUrl, $defaultSenderUrl, $insuranceRecalculateUrl, $adminCodRecalculateUrl] = $this->getAdminUrls();
        $countriesByIsoCode = $this->getCountries('iso_code', $this->context);
        $countriesByCountryCode= $this->getCountries('id_country', $this->context);
        $form = $this->createForm(ShippingFormType::class, null, ['countries' => $countriesByIsoCode]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $accessToken = $authGuardService->getAccessToken();
            $data = $form->getData();
            $courierData = $request->request->get('courier_form');
            $courierAdditionalSettingsData = $this->groupedAndFilterAdditionalSettings($request->request->get('additional_courier_settings_form'));
            $priceToFrontendData = $request->request->get('price_to_frontend_form');
            $defaultSenderData = $request->request->get('default_sender_form');
            $validationErrors = $this->validateAjaxFormData($courierData, $priceToFrontendData, $defaultSenderData);
            if (!empty($validationErrors)) {
                $this->addFlash('error', $this->formatValidationErrorMessage($validationErrors));
                return $this->render('@Modules/epaka/views/templates/admin/shipping/add.html.twig', [
                    'form' => $form->createView(),
                    'adminCouriersControllerUrl' => $couriersUrl,
                    'adminAdditionalCourierSettingsControllerUrl' => $additionalCourierSettingsUrl,
                    'adminPriceToFrontendControllerUrl' => $priceToFrontendUrl,
                    'adminDefaultSenderControllerUrl' => $defaultSenderUrl,
                    'adminInsuranceRecalculateUrl' => $insuranceRecalculateUrl,
                    'adminCodRecalculateUrl' => $adminCodRecalculateUrl,
                ]);
            }
            $priceToFrontendData['price'] = (float)str_replace(',', '.', trim($priceToFrontendData['price']));
            try {
                $apiCourierData = $this->getApiCourierData($accessToken, $data['height'], $data['length'], $data['weight'], $data['width'], $data['parcelType'], $data['shippingType'], $courierData['courier'], $data['receiverCountry'], $data['senderCountry']);
                $carrierImageUrl = $apiCourierData['logo'];
                $carrierTrackingUrl = '';
                $senderCountryId = $this->getCountryIdByIsoCode($data['senderCountry']);
                $receiverCountryId = $this->getCountryIdByIsoCode($data['receiverCountry']);
                $carrier = $this->createPrestashopCarrier($carrierTrackingUrl, $carrierImageUrl, $apiCourierData, $data, $priceToFrontendData, $senderCountryId, $receiverCountryId);
                $epakaCarrier = $this->createEpakaCarrier((int)$carrier->id, $apiCourierData['name'], isset($apiCourierData['courierDeliveryType']) && $apiCourierData['courierDeliveryType'] === 'point', $data, $courierData, $courierAdditionalSettingsData, $senderCountryId, $receiverCountryId);
                $epakaSender = $this->createEpakaSender($epakaCarrier->getId(), $defaultSenderData);
                $this->addFlash('success', 'Przewoźnik został pomyślnie utworzony');
                return $this->redirectToRoute('list_of_shipping');
            } catch (Exception $e) {
                LoggerService::logError(
                    $e->getMessage(),
                    $e->getCode(),
                    $e->getFile(),
                    $e->getLine()
                );
            }
        }
        return $this->render('@Modules/epaka/views/templates/admin/shipping/add.html.twig', [
            'form' => $form->createView(),
            'adminCouriersControllerUrl' => $couriersUrl,
            'adminAdditionalCourierSettingsControllerUrl' => $additionalCourierSettingsUrl,
            'adminPriceToFrontendControllerUrl' => $priceToFrontendUrl,
            'adminDefaultSenderControllerUrl' => $defaultSenderUrl,
            'adminInsuranceRecalculateUrl' => $insuranceRecalculateUrl,
            'adminCodRecalculateUrl' => $adminCodRecalculateUrl,
        ]);
    }
    /**
     * Validate form data submitted via AJAX
     * Returns array of error messages if validation fails, empty array if successful
     */
    private function validateAjaxFormData(?array $courierData, ?array $priceToFrontendData, ?array $defaultSenderData): array
    {
        $validationErrors = [];

        if (!is_array($courierData) || !isset($courierData['courier']) || empty($courierData['courier'])) {
            $validationErrors[] = 'Nie wybrano kuriera';
        }

        if (!is_array($priceToFrontendData) || !isset($priceToFrontendData['price']) || trim($priceToFrontendData['price']) === '') {
            $validationErrors[] = 'Nie wypełniono pola "Cena w sklepie"';
        }

        if (!is_array($defaultSenderData) || empty($defaultSenderData)) {
            $validationErrors[] = 'Nie wypełniono danych nadawcy';
        } else {
            // Validate individual required fields in sender data
            $requiredSenderFields = [
                'firstName' => 'Imię',
                'lastName' => 'Nazwisko',
                'companyName' => 'Nazwa sklepu nadawcy',
                'street' => 'Ulica',
                'houseNumber' => 'Numer domu',
                'postCode' => 'Kod pocztowy',
                'city' => 'Miasto',
                'phone' => 'Telefon',
                'email' => 'Email'
            ];

            foreach ($requiredSenderFields as $field => $label) {
                if (!isset($defaultSenderData[$field]) || trim($defaultSenderData[$field]) === '') {
                    $validationErrors[] = "Nie wypełniono pola \"{$label}\" w danych nadawcy";
                }
            }
        }

        return $validationErrors;
    }

    /**
     * Format validation errors for display
     */
    private function formatValidationErrorMessage(array $validationErrors): string
    {
        return 'Proszę poczekać na pełne załadowanie formularza i wypełnić wszystkie wymagane pola:<br><br>' .
               implode('<br>', array_map(function($error) {
                   return '• ' . $error;
               }, $validationErrors));
    }

    private function getAdminUrls(): array
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            $adminUrl = rtrim(Tools::getHttpHost(true), '/');
            return [
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaCouriers', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaAdditionalCourierSettings', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaPriceToFrontend', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaDefaultSender', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaInsuranceRecalculate', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaCodRecalculate', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaTestAction', true),
            ];
        } else {
            return [
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaCouriers', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaCouriers'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaAdditionalCourierSettings', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaAdditionalCourierSettings'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaPriceToFrontend', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaPriceToFrontend'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaDefaultSender', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaDefaultSender'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaInsuranceRecalculate', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaInsuranceRecalculate'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaCodRecalculate', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaCodRecalculate'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaTestAction', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaTestAction'),
            ];
        }
    }

    /**
     * @throws Exception
     */
    private function getApiCourierData(string $accessToken, string $height, string $length, string $weight, string $width, string $packageType, string $shippingType, string $courierId, string $receiverCountry, string $senderCountry): array
    {
        $packagesFromFrontend = [['height' => $height, 'length' => $length,'packageType' => $packageType,'weight' => $weight,'width' => $width]];
        try {
            $apiCourierRawData = ApiService::getCouriers($accessToken, $packagesFromFrontend, $shippingType, $senderCountry, $receiverCountry, $courierId);
            return current($this->normalizeCourierApiPackagesData($apiCourierRawData['couriers']));

        } catch (Exception $e) {
            throw new Exception(
                $e->getMessage(),
                $e->getCode()
            );
        }
    }
    private function createPrestashopCarrier(string $carrierTrackingUrl, string $carrierImageUrl, array $apiCourierData, array $mainData, ?array $priceToFrontendData, string $senderCountryId, string $receiverCountryId): Carrier
    {
        $multiplier = $this->getDimensionMultiplier();
        $carrier = new Carrier();
        $carrier->name = pSQL($mainData['name']);
        $carrier->active = (bool) $mainData['isDefaultActive'];
        $carrier->deleted = false;
        $carrier->is_module = true;
        $carrier->shipping_external = true;
        $cleanedString = pSQL(preg_replace('/[^a-z0-9\-_]/', '', strtolower($apiCourierData['name'])));
        $carrier->external_module_name = basename(dirname(__DIR__, 2));
        $carrier->need_range = true;
        $carrier->range_behavior = 0;
        $carrier->max_width = (int)($mainData['width'] * $multiplier);
        $carrier->max_height = (int)($mainData['height'] * $multiplier);
        $carrier->max_depth = (int)($mainData['length'] * $multiplier);
        $carrier->max_weight = (float)$mainData['weight'];
        $languages = Language::getLanguages(true);
        $carrier->url = $carrierTrackingUrl . $carrier->id;
        foreach ($languages as $language) {
            $carrier->delay[$language['id_lang']] = ' ';
        }
        if ($carrier->add()) {
            $rangeWeight = new RangeWeight();
            $rangeWeight->id_carrier = $carrier->id;
            $rangeWeight->delimiter1 = 0;
            $rangeWeight->delimiter2 = 9999999;
            $rangeWeight->add();
            $this->addCarrierToZones($carrier, $senderCountryId, $receiverCountryId, $priceToFrontendData['price'], $rangeWeight->id);
            $groups = Group::getGroups($this->context->language->id);
            $this->addCarrierToGroups($carrier, $groups);
            $this->createImageForCarrier($carrier, $carrierImageUrl, false);
        }
        $this->setDefaultTaxRuleForCarrier($carrier);

        return $carrier;
    }
    private function createEpakaCarrier(int $psCarrierId, string $epakaCourierName, bool $isPoint, array $mainData, array $courierData, array $courierAdditionalSettingsData, string $senderCountryId, string $receiverCountryId): EpakaCarrier
    {
        $em = $this->getDoctrine()->getManager();
        $epakaCarrier = new EpakaCarrier();
        $epakaCarrier->setPsCarrierId($psCarrierId);
        $epakaCarrier->setShipmentType(ShippingMapper::convertShippingTypeToInt(pSQL($mainData['shippingType'])));
        $epakaCarrier->setParcelType(ShippingMapper::convertParcelTypeToInt(pSQL($mainData['parcelType'])));
        $epakaCarrier->setSenderCountryId(pSQL($senderCountryId));
        $epakaCarrier->setReceiverCountryId(pSQL($receiverCountryId));
        $epakaCarrier->setAdditionalOptions($courierAdditionalSettingsData);
        $epakaCarrier->setEpakaCurierId(pSQL($courierData['courier']));
        $epakaCarrier->setEpakaCourierName(pSQL($epakaCourierName));
        $epakaCarrier->setIsPoint($isPoint);
        $epakaCarrier->setUpdatedAt(new DateTime());
        $epakaCarrier->setCreatedAt(new DateTime());
        $em->persist($epakaCarrier);
        $em->flush();

        return $epakaCarrier;
    }
    private function createEpakaSender(int $epakaCarrierId, array $defaultSenderData): EpakaSender
    {
        $em = $this->getDoctrine()->getManager();
        $epakaSender = new EpakaSender();
        $epakaSender->setCarrierId($epakaCarrierId);
        $epakaSender->setCompanyName(pSQL($defaultSenderData['companyName']));
        $epakaSender->setFirstName(pSQL($defaultSenderData['firstName']));
        $epakaSender->setLastName(pSQL($defaultSenderData['lastName']));
        $epakaSender->setCountryIsoCode(pSQL($defaultSenderData['country']));
        $epakaSender->setStreet(pSQL($defaultSenderData['street']));
        $epakaSender->setHouseNumber(pSQL($defaultSenderData['houseNumber']));
        $epakaSender->setPostalCode(pSQL($defaultSenderData['postCode']));
        $epakaSender->setCity(pSQL($defaultSenderData['city']));
        $epakaSender->setPhone(pSQL($defaultSenderData['phone']));
        $epakaSender->setEmail(pSQL($defaultSenderData['email']));
        $epakaSender->setCreatedAt(new DateTime());
        $em->persist($epakaSender);
        $em->flush();

        return $epakaSender;
    }
    private function getCountryIdByIsoCode(string $isoCode): ?int
    {
        $countryProvider = $this->getCountryDataProvider();
        $countries = $countryProvider->getCountries($this->context->language->id);
        foreach ($countries as $country) {
            if ($country['iso_code'] === $isoCode) {
                return $country['id_country'];
            }
        }
        return null;
    }
    private function setDefaultTaxRuleForCarrier(Carrier $carrier): void
    {
        $taxRuleGroupId = Db::getInstance()->getValue('
                SELECT `id_tax_rules_group` 
                FROM `'._DB_PREFIX_.'tax_rules_group` 
                WHERE name LIKE "%23%" 
                AND deleted = 0'
        );
        if ($taxRuleGroupId) {
            $carrier->setTaxRulesGroup((int)$taxRuleGroupId);
            Db::getInstance()->update('carrier',
                ['id_reference' => (int)$taxRuleGroupId],
                'id_carrier = ' . (int)$carrier->id
            );
        }
    }
    private function groupedAndFilterAdditionalSettings(?array $additionalSettings): array
    {
        // Return empty array if no settings provided
        if (!is_array($additionalSettings) || empty($additionalSettings)) {
            return [];
        }

        $result = [];
        $labels = [];
        foreach ($additionalSettings as $key => $value) {
            if (strpos($key, '_label') !== false) {
                $serviceKey = str_replace('_label', '', $key);
                $labels[$serviceKey] = $value;
                unset($additionalSettings[$key]);
            }
        }
        if (!empty($additionalSettings['collection']) && !empty($additionalSettings['collection_amount'])) {
            $result['collection'] = [
                'checkbox' => $additionalSettings['collection'],
                'amount' => $additionalSettings['collection_amount']
            ];
            if (isset($labels['collection'])) {
                $result['collection']['label'] = $labels['collection'];
            }
        }
        if (!empty($additionalSettings['insurance']) && !empty($additionalSettings['insurance_amount'])) {
            $result['insurance'] = [
                'checkbox' => $additionalSettings['insurance'],
                'amount' => $additionalSettings['insurance_amount']
            ];
            if (isset($labels['insurance'])) {
                $result['insurance']['label'] = $labels['insurance'];
            }
        }
        if (!empty($additionalSettings['cod']) && !empty($additionalSettings['cod_amount'])) {
            $result['cod'] = [
                'checkbox' => $additionalSettings['cod'],
                'amount' => $additionalSettings['cod_amount']
            ];
            if (isset($labels['cod'])) {
                $result['cod']['label'] = $labels['cod'];
            }
        }
        foreach ($additionalSettings as $key => $value) {
            if (!in_array($key, ['collection', 'collection_amount', 'insurance', 'insurance_amount', 'cod', 'cod_amount'])) {
                if ($value === '1' || $value === 1 || $value === true) {
                    $result[$key] = [
                        'value' => $value
                    ];
                    if (isset($labels[$key])) {
                        $result[$key]['label'] = $labels[$key];
                    }
                } else {
                    $result[$key] = $value;
                }
            }
        }

        return $result;
    }
    private function addCarrierToZones(Carrier $carrier, int $senderCountryId, int $receiverCountryId, float $price, int $rangeWeightId): void
    {
        $zones = Zone::getZones(true);
        foreach ($zones as $zone) {
            $countriesInZone = Country::getCountriesByZoneId($zone['id_zone'], $this->context->language->id);
            $countryIds = array_column($countriesInZone, 'id_country');
            if (in_array($senderCountryId, $countryIds) || in_array($receiverCountryId, $countryIds)) {
                $carrier->addZone($zone['id_zone']);
                Db::getInstance()->update('delivery',
                    [
                        'price' => $price,
                        'id_range_weight' => (int)$rangeWeightId
                    ],
                    'id_carrier = ' . (int)$carrier->id . ' AND id_zone = ' . (int)$zone['id_zone']
                );
            }
        }
    }
    private function addCarrierToGroups(Carrier $carrier, array $groups): void
    {
        foreach ($groups as $group) {
            Db::getInstance()->insert('carrier_group', [
                'id_carrier' => (int)$carrier->id,
                'id_group' => (int)$group['id_group'],
            ]);
        }
    }
    private function createImageForCarrier(Carrier $carrier, string $logoUrl, bool $isSSL = true): void
    {
        if(!$isSSL) {
            $context = stream_context_create([
                "ssl" => [
                    "verify_peer" => false,
                    "verify_peer_name" => false,
                ],
            ]);
            $logoContent = file_get_contents($logoUrl, false, $context);
        } else {
            $logoContent = file_get_contents($logoUrl);
        }
        if ($logoContent !== false) {
            $logoPath = $logoUrl;
        }
    }

    public function editAction(Request $request, int $id): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $em = $this->getDoctrine()->getManager();
        $epakaCarrierDb = $em->getRepository(EpakaCarrier::class)->findOneBy(['id' => $id]);
        $psCarrier = new Carrier($epakaCarrierDb->getPsCarrierId());
        [$couriersUrl, $additionalCourierSettingsUrl, $priceToFrontendUrl, $defaultSenderUrl, $insuranceRecalculateUrl, $adminCodRecalculateUrl] = $this->getAdminUrls();
        $formData = [
            'id' => $id,
            'name' => $psCarrier->name,
            'height' => $psCarrier->max_height,
            'length' => $psCarrier->max_depth,
            'width' => $psCarrier->max_width,
            'weight' => $psCarrier->max_weight,
            'shippingType' => ShippingMapper::convertShippingTypeToString($epakaCarrierDb->getShipmentType()),
            'parcelType' => ShippingMapper::convertParcelTypeToString($epakaCarrierDb->getParcelType()),
            'senderCountry' => $epakaCarrierDb->getSenderCountryId(),
            'receiverCountry' => $epakaCarrierDb->getReceiverCountryId(),
            'isDefaultActive' => (bool) $psCarrier->active
        ];
        $countriesByIsoCode = $this->getCountries('iso_code', $this->context);
        $form = $this->createForm(ShippingFormType::class, $formData, ['countries' => $countriesByIsoCode]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $accessToken = $authGuardService->getAccessToken();
            $data = $form->getData();
            $courierData = $request->request->get('courier_form');
            $courierAdditionalSettingsData = $this->groupedAndFilterAdditionalSettings($request->request->get('additional_courier_settings_form'));
            $priceToFrontendData = $request->request->get('price_to_frontend_form');
            $defaultSenderData = $request->request->get('default_sender_form');
            $validationErrors = $this->validateAjaxFormData($courierData, $priceToFrontendData, $defaultSenderData);
            if (!empty($validationErrors)) {
                $this->addFlash('error', $this->formatValidationErrorMessage($validationErrors));
                return $this->render('@Modules/epaka/views/templates/admin/shipping/edit.html.twig', [
                    'form' => $form->createView(),
                    'epakaCarrierId' => $epakaCarrierDb->getId(),
                    'epakaCurrentCourierId' => $epakaCarrierDb->getEpakaCurierId(),
                    'adminCouriersControllerUrl' => $couriersUrl,
                    'adminAdditionalCourierSettingsControllerUrl' => $additionalCourierSettingsUrl,
                    'adminPriceToFrontendControllerUrl' => $priceToFrontendUrl,
                    'adminDefaultSenderControllerUrl' => $defaultSenderUrl,
                    'adminInsuranceRecalculateUrl' => $insuranceRecalculateUrl,
                    'adminCodRecalculateUrl' => $adminCodRecalculateUrl,
                ]);
            }
            $priceToFrontendData['price'] = (float)str_replace(',', '.', trim($priceToFrontendData['price']));
            try {
                $apiCourierData = $this->getApiCourierData($accessToken, $data['height'], $data['length'], $data['weight'], $data['width'], $data['parcelType'], $data['shippingType'], $courierData['courier'], $data['receiverCountry'], $data['senderCountry']);
                $carrierImageUrl = $apiCourierData['logo'];
                $carrierTrackingUrl = '';
                $senderCountryId = $this->getCountryIdByIsoCode($data['senderCountry']);
                $receiverCountryId = $this->getCountryIdByIsoCode($data['receiverCountry']);
                $carrier = $this->updatePrestashopCarrier($carrierTrackingUrl, $carrierImageUrl, $apiCourierData, $data, $priceToFrontendData, $senderCountryId, $receiverCountryId, $epakaCarrierDb->getPsCarrierId());
                $epakaCarrierDb = $this->updateEpakaCarrier($id, $carrier->id, $apiCourierData['name'], isset($apiCourierData['courierDeliveryType']) && $apiCourierData['courierDeliveryType'] === 'point', $data, $courierData, $courierAdditionalSettingsData, $senderCountryId, $receiverCountryId);
                $epakaSender = $this->updateEpakaSender($epakaCarrierDb->getId(), $defaultSenderData);
                $this->addFlash('success', 'Przewoźnik został pomyślnie utworzony');
                return $this->redirectToRoute('list_of_shipping');
            } catch (Exception $e) {
                LoggerService::logError(
                    $e->getMessage(),
                    $e->getCode(),
                    $e->getFile(),
                    $e->getLine()
                );
            }
        }
        return $this->render('@Modules/epaka/views/templates/admin/shipping/edit.html.twig', [
            'form' => $form->createView(),
            'epakaCarrierId' => $epakaCarrierDb->getId(),
            'epakaCurrentCourierId' => $epakaCarrierDb->getEpakaCurierId(),
            'adminCouriersControllerUrl' => $couriersUrl,
            'adminAdditionalCourierSettingsControllerUrl' => $additionalCourierSettingsUrl,
            'adminPriceToFrontendControllerUrl' => $priceToFrontendUrl,
            'adminDefaultSenderControllerUrl' => $defaultSenderUrl,
            'adminInsuranceRecalculateUrl' => $insuranceRecalculateUrl,
            'adminCodRecalculateUrl' => $adminCodRecalculateUrl,
        ]);
    }
    private function updatePrestashopCarrier(string $carrierTrackingUrl, string $carrierImageUrl, array $apiCourierData, array $mainData, ?array $priceToFrontendData, string $senderCountryId, string $receiverCountryId, int $psCarrierId): Carrier
    {
        $multiplier = $this->getDimensionMultiplier();
        $carrier = new Carrier($psCarrierId);
        $carrier->force_id = true;
        $carrier->name = pSQL($mainData['name']);
        $carrier->active = (bool) $mainData['isDefaultActive'];
        $carrier->max_width = (int)((int)$mainData['width'] * $multiplier);
        $carrier->max_height = (int)((int)$mainData['height'] * $multiplier);
        $carrier->max_depth = (int)((int)$mainData['length'] * $multiplier);
        $carrier->max_weight = (float)$mainData['weight'];
        $languages = Language::getLanguages(true);
        foreach ($languages as $language) {
            $carrier->delay[$language['id_lang']] = ' ';
        }
        if ($carrier->update()) {
            $rangeWeightId = Db::getInstance()->getValue('
                SELECT id_range_weight 
                FROM '._DB_PREFIX_.'range_weight 
                WHERE id_carrier = '.(int)$carrier->id
            );

            if ($rangeWeightId) {
                $rangeWeight = new RangeWeight($rangeWeightId);
                $rangeWeight->delimiter2 = 9999999;
                $rangeWeight->update();
            } else {
                $rangeWeight = new RangeWeight();
                $rangeWeight->id_carrier = $carrier->id;
                $rangeWeight->delimiter1 = 0;
                $rangeWeight->delimiter2 = (float)$mainData['weight'];
                $rangeWeight->add();
                $rangeWeightId = $rangeWeight->id;
            }
            Db::getInstance()->delete('carrier_zone', 'id_carrier = ' . (int)$carrier->id);
            $this->addCarrierToZones($carrier, $senderCountryId, $receiverCountryId, $priceToFrontendData['price'], $rangeWeight->id);
            Db::getInstance()->delete('carrier_group', 'id_carrier = ' . (int)$carrier->id);
            $groups = Group::getGroups($this->context->language->id);
            $this->addCarrierToGroups($carrier, $groups);
            $this->createImageForCarrier($carrier, $carrierImageUrl, false);
        }
        $this->setDefaultTaxRuleForCarrier($carrier);

        return $carrier;
    }
    private function updateEpakaCarrier(int $id, int $psCarrierId, string $epakaCourierName, bool $isPoint, array $mainData, array $courierData, array $courierAdditionalSettingsData, string $senderCountryId, string $receiverCountryId): EpakaCarrier {
        $em = $this->getDoctrine()->getManager();
        $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['id' => $id]);
        $epakaCarrier->setPsCarrierId($psCarrierId);
        $epakaCarrier->setShipmentType(ShippingMapper::convertShippingTypeToInt(pSQL($mainData['shippingType'])));
        $epakaCarrier->setParcelType(ShippingMapper::convertParcelTypeToInt(pSQL($mainData['parcelType'])));
        $epakaCarrier->setSenderCountryId(pSQL($senderCountryId));
        $epakaCarrier->setReceiverCountryId(pSQL($receiverCountryId));
        $epakaCarrier->setAdditionalOptions($courierAdditionalSettingsData);
        $epakaCarrier->setEpakaCurierId(pSQL($courierData['courier']));
        $epakaCarrier->setEpakaCourierName(pSQL($epakaCourierName));
        $epakaCarrier->setIsPoint($isPoint);
        $epakaCarrier->setUpdatedAt(new DateTime());

        $em->flush();

        return $epakaCarrier;
    }
    private function updateEpakaSender(int $epakaCarrierId, array $defaultSenderData): EpakaSender {
        $em = $this->getDoctrine()->getManager();
        $epakaSender = $em->getRepository(EpakaSender::class)->findOneBy(['carrierId' => $epakaCarrierId]);


        $epakaSender->setCompanyName(pSQL($defaultSenderData['companyName']));
        $epakaSender->setFirstName(pSQL($defaultSenderData['firstName']));
        $epakaSender->setLastName(pSQL($defaultSenderData['lastName']));
        $epakaSender->setCountryIsoCode(pSQL($defaultSenderData['country']));
        $epakaSender->setStreet(pSQL($defaultSenderData['street']));
        $epakaSender->setHouseNumber(pSQL($defaultSenderData['houseNumber']));
        $epakaSender->setPostalCode(pSQL($defaultSenderData['postCode']));
        $epakaSender->setCity(pSQL($defaultSenderData['city']));
        $epakaSender->setPhone(pSQL($defaultSenderData['phone']));
        $epakaSender->setEmail(pSQL($defaultSenderData['email']));
        $em->flush();

        return $epakaSender;
    }

    public function deleteAction(Request $request, int $id): Response
    {
        /**
        * @var $epakaCarrier EpakaCarrier
        */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $em = $this->getDoctrine()->getManager();
        $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['id' => $id]);
        if($epakaCarrier) {
            $carrier = new Carrier($epakaCarrier->getPsCarrierId());
            if (Validate::isLoadedObject($carrier)) {
                $carrier->deleted = true;
                $carrier->update();
            }
            $epakaCarrier->setIsDeleted(true);
            $em->flush();
        }
        return $this->redirectToRoute('list_of_shipping');
    }

    /**
     * @throws Exception
     */
    public function couriersAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }


        $accessToken = $authGuardService->getAccessToken();
        $data = json_decode($request->getContent(), true);
        $currentEpakaCurierId = $data['currentEpakaCurierId'] ?? null;
        $packagesFromFrontend = $data['packages'] ?? [];
        $shippingType = $data['shippingType'] ?? null;
        $senderCountry = $data['senderCountry'] ?? null;
        $receiverCountry = $data['receiverCountry'] ?? null;
        try {
            $couriers = ApiService::getCouriers($accessToken, $packagesFromFrontend, $shippingType, $senderCountry, $receiverCountry);
        } catch (Exception $e) {
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
            $cleanErrorMessage = strip_tags($e->getMessage());
            $cleanErrorMessage = htmlspecialchars_decode($cleanErrorMessage, ENT_QUOTES);
            $cleanErrorMessage = str_replace("\\n", PHP_EOL, $cleanErrorMessage);
            return new JsonResponse(
                [
                    'message' => $cleanErrorMessage,
                ], 502
            );
        }
        $couriers = $this->normalizeCourierApiPackagesData($couriers['couriers'], $currentEpakaCurierId);
        $couriers = array_filter($couriers, function ($courier) {
           if($courier['name'] !== 'epaka.pl SPEDYCJA') return $courier;
        });
        if(empty($couriers)) return new JsonResponse(['message' => 'Nie znaleziono kurierów odpowiadających podanym parametrom'], 404);
        $form = $this->createForm(CourierFormType::class, null, [
            'couriers' => $couriers
        ]);
        return $this->render('@Modules/epaka/views/templates/admin/shipping/additionalFields/_courierFields.html.twig', [
            'form' => $form->createView()
        ]);
    }
    private function normalizeCourierApiPackagesData(array $couriers, int $currentActiveEpakaCourierId = null): array
    {
        return array_map(function($elem) use($currentActiveEpakaCourierId) {
            $elem['price'] = $elem['grossPriceTotal'];
            if(isset($elem['courier'])) {
                foreach ($elem['courier'] as $key => $value) {
                    $elem[$key] = $value;
                }
                unset($elem['courier']);
                $elem['isSelected'] = (int)$elem['id'] === $currentActiveEpakaCourierId;
                return $elem;
            }
        }, $couriers);
    }

    public function additionalCourierSettingsAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $em = $this->getDoctrine()->getManager();
        $authGuardService = $this->get('epaka.service.auth_guard');
        $accessToken = $authGuardService->getAccessToken();
        $data = json_decode($request->getContent(), true);
        $courierId = $data['courierId'] ?? null;
        $courierName = $data['courierName'] ?? null;
        $receiverCountry = $data['receiverCountry'] ?? null;
        $senderCountry = $data['senderCountry'] ?? null;
        $shippingType = $data['shippingType'] ?? null;
        $receiverPostCode = $data['receiverPostCode'] ?? null;
        $senderPostCode = $data['senderPostCode'] ?? null;
        $psEpakaCarrierId = $data['psEpakaCarrierId'] ?? null;

        $existingAdditionalFieldsData = null;
        if($psEpakaCarrierId) {
            $epakaCarrier =  $em->getRepository(EpakaCarrier::class)->findOneBy(['id' => $psEpakaCarrierId]);
            $existingAdditionalFieldsData = $epakaCarrier->getAdditionalOptions();
        }
        try {
            $courierAdditionalFields = ApiService::getServices(
                $authGuardService->getAccessToken(),
                (int) $courierId,
                $senderCountry,
                $receiverCountry,
                $shippingType,
                $senderPostCode,
                $receiverPostCode,
                $existingAdditionalFieldsData
            );
        } catch (Exception $e) {
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
        }
        $additionalForm = $this->createForm(AdditionalCourierSettingsFormType::class, null, [
            'options' => $courierAdditionalFields
        ]);

        return $this->render('@Modules/epaka/views/templates/admin/shipping/additionalFields/_courierAdditionalSettingsFields.html.twig', [
            'form' => $additionalForm->createView()
        ]);
    }

    public function priceToFrontendAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $em = $this->getDoctrine()->getManager();
        $data = json_decode($request->getContent(), true);
        $epakaCarrierId = $data['epakaCarrierId'] ?? null;
        $formData = null;
        if($epakaCarrierId) {
            $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['id' => $epakaCarrierId]);
            $psCarrier = new Carrier($epakaCarrier->getPsCarrierId());
            $prices = $this->getCarrierZonesData($psCarrier->id);
            $formData = [
                'price' => $prices ? round($prices[0]['price'], 2) : 0
            ];
        }
        $form = $this->createForm(PriceToFrontendFormType::class, $formData);
        return $this->render('@Modules/epaka/views/templates/admin/shipping/additionalFields/_priceToFrontendFields.html.twig', [
            'form' => $form->createView()
        ]);
    }

    public function defaultSenderAction(Request $request): Response
    {
        /**
        * @var $epakaSender EpakaSender
        * @var $epakaSettings EpakaSettings
        */
        $data = json_decode($request->getContent(), true);
        $epakaCarrierId = $data['epakaCarrierId'] ?? null;
        $em = $this->getDoctrine()->getManager();
        $countriesByIsoCode = $this->getCountries('iso_code', $this->context);
        $formData = null;
        $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
        switch (true) {
            case $epakaCarrierId: {
                $epakaSender = $em->getRepository(EpakaSender::class)->findOneBy(['carrierId' => $epakaCarrierId]);
                if ($epakaSender) {
                    $formData = [
                        'firstName' => $epakaSender->getFirstName(),
                        'lastName' => $epakaSender->getLastName(),
                        'companyName' => $epakaSender->getCompanyName(),
                        'street' => $epakaSender->getStreet(),
                        'houseNumber' => $epakaSender->getHouseNumber(),
                        'postCode' => $epakaSender->getPostalCode(),
                        'country' => $epakaSender->getCountryIsoCode(),
                        'city' => $epakaSender->getCity(),
                        'phone' => $epakaSender->getPhone(),
                        'email' => $epakaSender->getEmail(),
                    ];
                }
                break;
            }
            case isset($epakaSettings): {
                $formData = [
                    'firstName' => $epakaSettings->getFirstName(),
                    'lastName' => $epakaSettings->getLastName(),
                    'companyName' => $epakaSettings->getCompanyName(),
                    'street' => $epakaSettings->getStreet(),
                    'houseNumber' => $epakaSettings->getHouseNumber(),
                    'postCode' => $epakaSettings->getPostCode(),
                    'country' => $epakaSettings->getCountry(),
                    'city' => $epakaSettings->getCity(),
                    'phone' => $epakaSettings->getPhone(),
                    'email' => $epakaSettings->getEmail(),
                ];
            }
                break;
        }
        $form = $this->createForm(DefaultSenderFormType::class, $formData, [
            'countries' => $countriesByIsoCode,
            'selectCountry' => $formData['country'] ?? 'PL'
        ]);
        return $this->render('@Modules/epaka/views/templates/admin/shipping/additionalFields/_defaultSenderFields.html.twig', [
            'form' => $form->createView()
        ]);
    }

    public function insuranceRecalculateAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $accessToken = $authGuardService->getAccessToken();
        $data = json_decode($request->getContent(), true);
        $courierId = $data['courierId'] ?? [];
        $insuranceValue = $data['declaredValue'] ?? '';
        $maxWeight = $data['maxWeight'] ?? '';
        $receiverCountry = $data['receiverCountry'] ?? '';
        $senderCountry = $data['senderCountry'] ?? '';
        try {
            $insurance = ApiService::getInsurancePrice(
                $accessToken,
                (int) $courierId,
                $senderCountry,
                $receiverCountry,
                $maxWeight,
                $insuranceValue
            );
            return new Response(json_encode($insurance));
        } catch (Exception $e) {
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
            return new Response(json_encode(['price' => 0]));
        }
    }

    public function codRecalculateAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $accessToken = $authGuardService->getAccessToken();
        $data = json_decode($request->getContent(), true);
        $amount = $data['amount'] ?? null;
        $courierId = $data['courierId'] ?? null;
        try {
            $cod = ApiService::getCodPrice(
                $accessToken,
                (int) $courierId,
                (float) $amount
            );
            return new Response(json_encode(
                is_array($cod) && isset( $cod['codSaldo']) ? ['price' => (float)$cod['codSaldo']] :  ['price' => current($cod)]
            ));
        } catch (Exception $e) {
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
            return new JsonResponse([
                'message' => $e->getMessage(),
            ], $e->getCode());
        }
    }

    private function getCarrierZonesData(int $psCarrierId): ?array
    {
        return Db::getInstance()->executeS('
                SELECT d.*, z.name as zone_name
                FROM `'._DB_PREFIX_.'delivery` d
                LEFT JOIN `'._DB_PREFIX_.'zone` z ON (z.id_zone = d.id_zone)
                WHERE d.id_carrier = '. $psCarrierId
        );
    }
    private function getDimensionMultiplier()
    {
        // Get PrestaShop's configured dimension unit
        $dimensionUnit = Configuration::get('PS_DIMENSION_UNIT');

        // Since input is in centimeters, we need different multipliers
        // to convert from centimeters to target unit
        $conversion = [
            'cm' => 1,     // Keep as is (but will be converted to int later)
            'mm' => 10,    // Convert cm to mm (0.5 cm * 10 = 5 mm)
            'in' => 0.3937 // Convert cm to inches
        ];

        // If unit not found, assume centimeters
        return $conversion[$dimensionUnit] ?? 1;
    }
}
