<?php

namespace Epaka\Controller;

use Carrier;
use Context;
use DateTime;
use Epaka\Entity\EpakaCarrier;
use Epaka\Entity\EpakaOrder;
use Epaka\Entity\EpakaParcel;
use Epaka\Entity\EpakaSettings;
use Epaka\Mapper\CourierStatusMapper;
use Epaka\Mapper\DocumentsLabelFormatMapper;
use Epaka\Mapper\EpakaOrderStatusMapper;
use Epaka\Service\ApiService;
use Epaka\Service\EpakaOrderStatusUpdateService;
use Epaka\Service\LoggerService;
use Epaka\Service\PdfService;
use Exception;
use Order;
use OrderState;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tools;

class ShipmentController extends FrameworkBundleAdminController
{
    private $context;

    public function __construct()
    {
        parent::__construct();
        $this->context = Context::getContext();
    }

    public function listAction(Request $request): Response
    {
        /**
         * @var $epakaParcel EpakaOrder;
         * @var $epakaCarrier EpakaCarrier;
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        $em = $this->getDoctrine()->getManager();
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        [$orderAdditionalInfoUrl, $orderApiCreateUrl] = $this->getAdminUrls();
        $isDeleteView = (bool)$request->query->get('isDeleteView', 0);
        $epakaParcels = $em->getRepository(EpakaParcel::class)->findBy(['isDeleted' => $isDeleteView], ['id' => 'DESC']);
        foreach ($epakaParcels as $epakaParcel) {
            $isEpakaCancelStatus = EpakaOrderStatusMapper::CANCELED === $epakaParcel->getEpakaStatus();
            if($isEpakaCancelStatus && !$epakaParcel->isDeleted()) {
                $epakaParcel->setIsDeleted(true);
                $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
                $em->flush();
            }
        }
        if (!$isDeleteView) {
            $epakaParcels = array_filter($epakaParcels, function ($parcel) {return !$parcel->isDeleted();});
        }
        $parcels = [];
        foreach ($epakaParcels as $epakaParcel) {
            $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaParcel->getEpakaOrderId()]);
            $psOrder = new Order($epakaOrder->getPsOrderId());
            $orderStateId = $psOrder->getCurrentState();
            $orderState = new OrderState($orderStateId);
            $langId = (int)$this->context->language->id;
            $orderStatusName = $orderState->name[$langId] ?? 'Unknown';
            $statusColor = $orderState->color;
            $psCarrier = new Carrier($psOrder->id_carrier);
            $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['psCarrierId' => $psCarrier->id]);
            $orderProductNamesAndQty = '-';
            $orderCreatedAt = new DateTime($psOrder->date_add);
            foreach ($psOrder->getProducts() as $product) {
                $orderProductNamesAndQty = "(" . $product['product_quantity'] . ") " . $product['product_name'];
            }
            $parcels[] = [
                'id' => $epakaParcel->getId(),
                'epakaApiId' => $epakaParcel->getApiOrderId(),
                'epakaOrderId' => $epakaParcel->getEpakaOrderId(),
                'waybillNumber' => $epakaParcel->getWaybillNumber() ?? 'Oczekiwanie',
                'courierName' => isset($epakaCarrier)? $epakaCarrier->getEpakaCourierName() : '-',
                'orderStatusBadgeName' => EpakaOrderStatusMapper::getBadgeNameByIntStatus($epakaParcel->getEpakaStatus()),
                'orderStatus' => $epakaParcel->getEpakaStatus() !== null
                    ? EpakaOrderStatusMapper::convertIntToHumanReadableStatus($epakaParcel->getEpakaStatus())
                    : EpakaOrderStatusMapper::convertIntToHumanReadableStatus(EpakaOrderStatusMapper::UNKNOWN),
                'courierStatusBadgeName' => CourierStatusMapper::getBadgeNameByIntStatus($epakaParcel->getCourierStatus()),
                'courierStatus' => $epakaParcel->getCourierStatus() !== null
                    ? CourierStatusMapper::convertIntToHumanReadableStatus($epakaParcel->getCourierStatus())
                    : CourierStatusMapper::convertIntToHumanReadableStatus(CourierStatusMapper::PACKAGE_NO_SEND),
                'createdAt' => $epakaParcel->getEpakaPlacedAt()->format('d/m/Y'),
                'isDeleted' => $epakaParcel->isDeleted()
            ];
        }
        return $this->render($isDeleteView ? '@Modules/epaka/views/templates/admin/shipment/deleted_list.html.twig' : '@Modules/epaka/views/templates/admin/shipment/list.html.twig', [
            'parcels' => $parcels,
            'epakaOrderStatuses' => EpakaOrderStatusMapper::HUMAN_READABLE_STATUSES,
            'courierStatuses' => CourierStatusMapper::HUMAN_READABLE_STATUSES,
            'orderAdditionalInfoUrl' => $orderAdditionalInfoUrl,
            'orderApiCreateUrl' => $orderApiCreateUrl,
            'isShipmentPage' => true
        ]);
    }

    public function updateStatusesAction(): Response
    {

        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        try {
            EpakaOrderStatusUpdateService::updateStatuses();
            return $this->redirectToRoute('list_of_shipment');
        } catch (Exception $e) {
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
            return $this->redirectToRoute('list_of_shipment');
        }
        return $this->redirectToRoute('list_of_shipment');
    }

    public function massAction(Request $request): Response
    {
        /**
        * @var $epakaParcel EpakaParcel
        */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $massEpakaDeleteType = 'delete-shipments';
        $massEpakaPrintLabelsType = 'bulk-label-printing';
        $massType = $request->request->get('massType');
        $recordsIds = json_decode($request->request->get('recordsIds'), true) ?? [];
        $em = $this->getDoctrine()->getManager();
        $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
        $rowErrors = [];
        $errorFlashMessage = null;
        $errorFlashMessageType= null;
        $recordsIds = array_values(array_filter($recordsIds));
        if($massType === $massEpakaPrintLabelsType) {
            if (empty($recordsIds)) {
                $this->addFlash('warning', 'Wybierz co najmniej jedną przesyłkę do drukowania etykiet');
                return $this->redirectToRoute('list_of_shipment');
            }
            $tempDir = sys_get_temp_dir();
            $tempFiles = [];
            foreach ($recordsIds as $recordId) {
                $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['id' => (int) $recordId]);
                if (!$epakaParcel) continue;
                try {
                    $type = DocumentsLabelFormatMapper::convertIntToStringStatus($epakaSettings->getDefaultLabelFormat());
                    $orderDocumentInBase64 = ApiService::fetchOrderDocument($authGuardService->getAccessToken(), $epakaParcel->getApiOrderId(), $type);
                    if (is_array($orderDocumentInBase64) && isset($orderDocumentInBase64['document'])) {
                        $pdfContent = base64_decode($orderDocumentInBase64['document']);
                        $tempFile = $tempDir . '/epaka_label_' . $recordId . '.pdf';
                        file_put_contents($tempFile, $pdfContent);
                        $tempFiles[] = $tempFile;
                    }
                } catch (Exception $e) {
                    LoggerService::logError(
                        $e->getMessage(),
                        $e->getCode(),
                        $e->getFile(),
                        $e->getLine()
                    );
                    $errorsRaw = json_decode($e->getMessage(), true);
                    $error = LoggerService::normalizeErrorToCollection((int)$recordId, $errorsRaw, 'Błąd podczas pobierania dokumentu');
                    $rowErrors[] = ['id' => $error['id'], 'message' => $error['message']];
                }
            }
            if(empty($tempFiles)) {
                $this->addFlash('warning', 'Pliki do scalenia nie zostały określone');
                return $this->redirectToRoute('list_of_shipment');
            }
            try {
                PdfService::mergePdfFiles($tempFiles, 'etykiety-zbiorcze.pdf');
            } catch (Exception $e) {
                LoggerService::logError(
                    'PDF Merge Error: ' . $e->getMessage(),
                    $e->getCode(),
                    $e->getFile(),
                    $e->getLine()
                );
                $this->addFlash('warning', 'PDF Merge Error: ' . $e->getMessage());
                return $this->redirectToRoute('list_of_shipment');
            }
            $flashError = LoggerService::normalizeCollectionFlashMessages($rowErrors, 'Wystąpił problem podczas łączenia etykiet', 'Wszystkie etykiety zostały pomyślnie połączone', 'error');
            if($flashError) {
                $errorFlashMessage = $flashError['message'];
                $errorFlashMessageType = $flashError['type'];
            }

        } else if($massType === $massEpakaDeleteType) {
            foreach ($recordsIds as $recordId) {
                $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['id' => (int) $recordId]);
                if(!$epakaParcel) {
                    LoggerService::logError(
                        'EpakaOrder instance not found with ID: ' . $recordId,
                        500,
                        __FILE__,
                        "(EpakaOrder instance not found while creating an order on Epaka)"
                    );
                    $rowErrors[] = ['id' => $recordId, ' message' => 'EpakaOrder instance not found'];
                }

                try {
                    ApiService::cancelOrder($authGuardService->getAccessToken(), $epakaParcel->getApiOrderId());
                    $epakaParcel->setIsDeleted(true);
                    $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
                    $em->flush();
                } catch (Exception $e) {
                    if ($e->getCode() === 409) {
                        $this->addFlash('warning', 'Zamówienie już zostało anulowane.');
                        $epakaParcel->setIsDeleted(true);
                        $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
                        $em->flush();
                    } else {
                        LoggerService::logError(
                            $e->getMessage(),
                            $e->getCode(),
                            $e->getFile(),
                            $e->getLine()
                        );
                        $errorsRaw = json_decode($e->getMessage(), true);
                        $error = LoggerService::normalizeErrorToCollection((int)$recordId, $errorsRaw, 'Błąd podczas usuwania przesyłki');
                        $rowErrors[] = ['id' => $error['id'], 'message' => $error['message']];
                    }
                }
            }
            $flashError = LoggerService::normalizeCollectionFlashMessages($rowErrors, 'Wystąpiły błędy podczas usuwania przesyłek', 'Wszystkie przesyłki zostały pomyślnie usunięte', 'error');
            if($flashError) {
                $errorFlashMessage = $flashError['message'];
                $errorFlashMessageType = $flashError['type'];
            }
        }

        if($errorFlashMessageType && $errorFlashMessage) {
            $this->addFlash($errorFlashMessageType, $errorFlashMessage);
        }
        return $this->redirectToRoute('list_of_shipment');
    }

    private function getAdminUrls(): array
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            $adminUrl = rtrim(Tools::getHttpHost(true), '/');
            return [
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaOrderAdditionalInfo', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaOrderApiCreate', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaShipmentApiMass', true),
            ];
        } else {
            return [
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderAdditionalInfo', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaCouriers'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderApiCreate', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaOrderApiCreate'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaShipmentApiMass', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaShipmentApiMass'),
            ];
        }
    }
}