<?php

namespace Epaka\Controller;

use Context;
use Epaka\Entity\EpakaAuth;
use Epaka\Entity\EpakaSettings;
use Epaka\Form\SettingsFormType;
use Epaka\Mapper\DocumentsLabelFormatMapper;
use Epaka\Mapper\PaymentTypeMapper;
use Epaka\Service\ApiService;
use Epaka\Service\CryptoService;
use Epaka\Service\LoggerService;
use Epaka\Traits\CountryTrait;
use Exception;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;

class SettingsController extends FrameworkBundleAdminController
{
    use CountryTrait;
    public function __construct()
    {
        parent::__construct();
    }

    public function settingsAction(Request $request): Response
    {
        /**
         * @var $epakaSettings EpakaSettings
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $em = $this->getDoctrine()->getManager();
        try {
            $accountBalance = ApiService::fetchAccountBalance($authGuardService->getAccessToken());
        } catch (Exception $e) {
            $accountBalance = null;
            $this->addFlash('error', 'Coś poszło nie tak. Nie udało się pobrać salda Twojego konta. Prosimy o sprawdzenie ustawień konta i kontakt z epaka.pl w celu uzyskania pomocy.');
            LoggerService::logError(
                $e->getMessage(),
                $e->getCode(),
                $e->getFile(),
                $e->getLine()
            );
        }
        try {
            $epakaUser = ApiService::fetchUserInfo($authGuardService->getAccessToken());
        } catch (Exception $e) {
            $errorRaw = json_decode($e->getMessage(), true);
            $error = LoggerService::normalizeErrorToFlashMessage($errorRaw, 'Błąd pobierania danych użytkownika');
            LoggerService::logError(
                $error['message'],
                $e->getCode() ?? 0,
                $e->getFile(),
                $e->getLine()
            );
        }
        $userEmail = $epakaUser['email'] ?? '';
        $paymentTypes = PaymentTypeMapper::HUMAN_READABLE_STATUSES;
        if($epakaUser['paymentType'] == 'gotówkowy') unset($paymentTypes[PaymentTypeMapper::CUMULATIVE_PAYMENT_TYPE]);
        $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
        $countriesByIsoCode = $this->getCountries('iso_code', $this->getContext());
        $form = $this->createForm(SettingsFormType::class, $epakaSettings,
        [
            'userEmail' => $userEmail,
            'defaultLabelFormatChoices' => DocumentsLabelFormatMapper::HUMAN_READABLE_STATUSES,
            'paymentTypes' => $paymentTypes,
            'countries' => $countriesByIsoCode
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            switch (true) {
                case $form->get('connectToEpakaAccountButton')->isClicked(): {
                    $authRecord = $em->getRepository(EpakaAuth::class)->findOneBy([]) ?? null;
                    if($authRecord) {
                        $em->remove($authRecord);
                        $em->flush();
                        $this->addFlash('success', 'Twoje konto epaka zostało pomyślnie odłączone.');
                        return $this->redirectToRoute('auth');
                    }
                    break;
                }
                case $form->get('send')->isClicked(): {
                    $epakaSettings = $form->getData();
                    $epakaSettings->setDefaultLabelFormat($epakaSettings->getDefaultLabelFormat());
                    $epakaSettings->setPaymentType($epakaSettings->getPaymentType());
                    $em->flush();
                    $this->addFlash('success', 'Twoje ustawienia zostały pomyślnie zmienione.');
                    break;
                }
            }
        }
        $siteUrl = Context::getContext()->shop->getBaseURL(true, true);
        $cronToken = CryptoService::encryptStatic($siteUrl, 'bC15HarSft');
        return $this->render('@Modules/epaka/views/templates/admin/setting/settings.html.twig', [
            'form' => $form->createView(),
            'accountBalance' => $accountBalance,
            'cronUrl' => $siteUrl . 'module/epaka/cron?token=' . $cronToken
        ]);
    }
}