<?php
namespace Epaka\Controller;

use Address;
use Carrier;
use Cart;
use Configuration;
use Context;
use Country;
use Customer;
use DateTime;
use Db;
use Epaka\Entity\EpakaAuth;
use Epaka\Entity\EpakaCarrier;
use Epaka\Entity\EpakaOrder;
use Epaka\Entity\EpakaParcel;
use Epaka\Entity\EpakaSender;
use Epaka\Entity\EpakaSettings;
use Epaka\Mapper\CourierStatusMapper;
use Epaka\Mapper\DocumentsLabelFormatMapper;
use Epaka\Mapper\EpakaOrderProcessResultStatusResultMapper;
use Epaka\Mapper\EpakaOrderStatusMapper;
use Epaka\Mapper\PaymentTypeMapper;
use Epaka\Mapper\ShippingMapper;
use Epaka\Service\ApiFieldLengthService;
use Epaka\Service\ApiOrderService;
use Epaka\Service\ApiService;
use Epaka\Service\LoggerService;
use Epaka\Tests\TestOrder;
use Exception;
use Order;
use OrderHistory;
use OrderState;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use Product;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tools;
use PrestaShop\PrestaShop\Adapter\Validate;

class OrderController extends FrameworkBundleAdminController
{
    public const DHL_API_ID = 8;
    public const DHL_SERVICE_POINT_API_ID = 22;
    public const DHL_TYPE_WEIGHT = 31.5;
    public const UPS_STANDARD = 'UPS Standard';
    public const UPS_EXPRESS_SAVER = 'UPS Express Saver';
    public const UPS_EXPRESS = 'UPS Express';
    public const UPS_EXPRESS_PLUS = 'UPS Express Plus';

    private $context;

    public function __construct()
    {
        parent::__construct();
        $this->context = Context::getContext();
    }

    public function list(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        $em = $this->getDoctrine()->getManager();
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        [$orderAdditionalInfoUrl, $orderApiCreateUrl, $orderApiMassUrl] = $this->getAdminUrls();
        $rawOrdersData = $em->getRepository(EpakaOrder::class)
            ->findBy([], ['createdAt' => 'DESC']);
        $orders = [];
        /**
         * @var $epakaOrder EpakaOrder;
         * @var $epakaCarrier EpakaCarrier;
         * @var $pakaParcel EpakaParcel;
         */
        foreach ($rawOrdersData as $epakaOrder) {
            $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['epakaOrderId' => $epakaOrder->getId(), 'isDeleted' => false]);
            $psOrder = new Order($epakaOrder->getPsOrderId());
            $orderStateId = $psOrder->getCurrentState();
            $orderState = new OrderState($orderStateId);
            $langId = (int)$this->context->language->id;
            $orderStatusName = $orderState->name[$langId] ?? 'Unknown';
            $statusColor = $orderState->color;
            $psCarrier = new Carrier($psOrder->id_carrier);
            $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['psCarrierId' => $psCarrier->id]);
            $orderProductNamesAndQty = '-';
            $orderCreatedAt = new DateTime($psOrder->date_add);
            $productsArray = [];
            foreach ($psOrder->getProducts() as $product) {
                $productsArray[] = "(".$product['product_quantity'].") ".$product['product_name']."<br>";
            }
            $orderProductNamesAndQty = !empty($productsArray) ? rtrim(implode('', $productsArray), '<br>') : '-';
            $address = new Address($psOrder->id_address_delivery);
            $country = new Country($address->id_country);
            $orders[] = [
                'id' => $epakaOrder->getId(),
                'epakaApiId' => $epakaParcel ? $epakaParcel->getApiOrderId() : null,
                'productNamesAndQty' => $orderProductNamesAndQty,
                'totalPaidTaxIncl' => $psOrder->total_paid_tax_incl,
                'isHasInvoice' => $this->getIsOrderHasInvoice($psOrder),
                'statusInText' => $epakaParcel && $epakaParcel->getOrderProcessResultStatus() !== null && !$epakaParcel->isDeleted() ? EpakaOrderProcessResultStatusResultMapper::convertIntToHumanReadableStatus($epakaParcel->getOrderProcessResultStatus()) : null,
                'statusBadgeName' => $epakaParcel && $epakaParcel->getOrderProcessResultStatus() !== null && !$epakaParcel->isDeleted() ? EpakaOrderProcessResultStatusResultMapper::getBadgeNameByIntStatus($epakaParcel->getOrderProcessResultStatus()) : null,
                'defaultPsCoreStatusLabel' => $orderStatusName ?? null,
                'defaultPsCoreStatusColor' => $statusColor ?? null,
                'carrierName' => isset($epakaCarrier)? $epakaCarrier->getEpakaCourierName() : '-',
                'templateName' => $psCarrier->name,
                'apiParcelNumber' => $epakaParcel ? $epakaParcel->getApiOrderId() : '-',
                'receiver' => [
                    'firstName' => ucfirst($address->firstname),
                    'lastName' => ucfirst($address->lastname),
                    'company' => $address->company ?: '-',
                    'address1' => $address->address1 ?: '-',
                    'address2' => $address->address2 ?: '-',
                    'city' => $address->city ?: '-',
                    'state' => $address->id_state ? (new State($address->id_state))->name : '-',
                    'postcode' => $address->postcode ?: '-',
                    'country' => $country->name[$langId],
                    'phone' => $address->phone ?: '-',
                    'email' => (new Customer($psOrder->id_customer))->email ?: '-'
                ],
                'createdAt' => $orderCreatedAt->format('d/m/Y'),
            ];
        }
        return $this->render('@Modules/epaka/views/templates/admin/order/list.html.twig', [
            'orders' => $orders,
            'psCoreOrderStatuses' => $this->getAllOrderStatuses(),
            'orderAdditionalInfoUrl' => $orderAdditionalInfoUrl,
            'orderApiCreateUrl' => $orderApiCreateUrl,
            'orderApiMassUrl' => $orderApiMassUrl,
            'isShipmentPage' => false
        ]);
    }

    public function additionalInfoAction(Request $request): Response
    {
        /**
         * @var $epakaOrder EpakaOrder
         * @var $epakaCarrier EpakaCarrier
         * @var $epakaSender EpakaSender
         * @var $epakaSettings EpakaSettings
         * @var $epakaParcel EpakaParcel
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $em = $this->getDoctrine()->getManager();
        $data = json_decode($request->getContent(), true);
        $epakaOderIdOrParcerlId = $data['epakaOderId'] ?? null;
        $isShipmentPage = $data['isShipmentPage'] ?? null;
        $epakaParcel = $isShipmentPage
            ? $em->getRepository(EpakaParcel::class)->findOneBy(['id' => $epakaOderIdOrParcerlId, 'isDeleted' => false])
            : $em->getRepository(EpakaParcel::class)->findOneBy(['epakaOrderId' => $epakaOderIdOrParcerlId, 'isDeleted' => false]);
        $epakaOrder = isset($epakaParcel)
            ? $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaParcel->getEpakaOrderId()])
            : $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaOderIdOrParcerlId]);
        $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
        if($isShipmentPage && !$epakaOrder && $epakaOderIdOrParcerlId) {
            $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['id' => $epakaOderIdOrParcerlId, 'isDeleted' => true]);
            if($epakaParcel->getEpakaOrderId()) {
                $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaParcel->getEpakaOrderId()]);
            }
        }
        if(!$epakaOrder) {
            return new JsonResponse([
                'message' => 'Wystąpił błąd podczas pobierania szczegółowych informacji o zamówieniu.',
            ], 404);
        }
        $psOrder = new Order($epakaOrder->getPsOrderId());
        $trackingNumber = Db::getInstance()->getValue('
            SELECT tracking_number
            FROM `' . _DB_PREFIX_ . 'order_carrier`
            WHERE id_order = ' . (int)$psOrder->id
        );
        if (!$trackingNumber) $trackingNumber = null;
        $psCarrier = new Carrier($psOrder->id_carrier);
        $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['psCarrierId' => $psCarrier->id]);
        $epakaSender = $em->getRepository(EpakaSender::class)->findOneBy(['carrierId' => $epakaCarrier->getId()]);

        $customer = new Customer($psOrder->id_customer);
        $customerAddress = new Address($psOrder->id_address_invoice);
        $receiverAddress = new Address($psOrder->id_address_delivery);
        $invoiceAddress = new Address($psOrder->id_address_invoice);
        $senderCountryIso = isset($epakaSender) ? $epakaSender->getCountryIsoCode() : '';
        $receiverCountryIso = (new Country($customerAddress->id_country))->iso_code;
        $orderAdditionalOptions = [];
        $additionalOptions = $epakaCarrier->getAdditionalOptions();
        if ($additionalOptions && (is_array($additionalOptions) || is_object($additionalOptions))) {
            foreach ($additionalOptions as $key => $option) {
                $orderAdditionalOptions[] = [
                    'name' => $key,
                    'label' => $option['label'] ?? ucfirst(str_replace('_', ' ', $key))
                ];
            }
        }
        $senderCountryIso = $epakaSender->getCountryIsoCode();
        $countryId = Db::getInstance()->getValue('
            SELECT id_country 
            FROM `' . _DB_PREFIX_ . 'country` 
            WHERE iso_code = "' . pSQL($senderCountryIso) . '"
        ');
        $psOrder->gift_message = '';
        [$totalLength, $totalWidth, $totalHeight] =$this->getOrderPackageDimensions($psOrder);
        $this->getLinkToOrderPsCorePage($this->context, (int) $psOrder->id);
        $invoiceData = [
            'firstName' => $invoiceAddress->firstname,
            'lastName' => $invoiceAddress->lastname,
            'company' => $invoiceAddress->company ?: '—',
            'vatNumber' => $invoiceAddress->vat_number ?: '—',
            'address' => $invoiceAddress->address1,
            'city' => $invoiceAddress->city,
            'postCode' => $invoiceAddress->postcode,
            'country' => Country::getNameById($psOrder->id_lang, $invoiceAddress->id_country),
            'phone' => $invoiceAddress->phone ?: '—',
            'email' => $customer->email,
        ];

        $additionalInfo = [
            'isCreated' => $epakaParcel instanceof EpakaParcel,
            'isPsCanceled' => $this->isOrderCanceled($psOrder),
            'customer' => [
                'firstName' => ucfirst($customer->firstname),
                'lastName' => ucfirst($customer->lastname),
                'phone' => $customerAddress->phone ?: $customerAddress->phone_mobile,
                'email' => $customer->email ?: null,
                'city' => $customerAddress->city ?: null,
                'country' => Country::getNameById($psOrder->id_lang, $customerAddress->id_country),
                'postCode' => $customerAddress->postcode ?: null,
            ],
            'recipient' => [
                'firstName' => $receiverAddress->firstname ? ucfirst($receiverAddress->firstname): null,
                'lastName' => $receiverAddress->lastname ? ucfirst($receiverAddress->lastname): null,
                'address1' => $customerAddress->address1 ?: null,
                'address2' => $customerAddress->address2 ?: null,
                'postCode' => $customerAddress->postcode ?: null,
                'city' => $customerAddress->city ?: null,
                'country' => Country::getNameById($psOrder->id_lang, $receiverAddress->id_country),
                'phone' => $receiverAddress->phone ?: $receiverAddress->phone_mobile,
                'email' => $customer->email ?: null,
            ],
            'sender' => [
                'companyName' => $epakaSender->getCompanyName() ?: null,
                'firstName' => $epakaSender->getFirstName() ? ucfirst($epakaSender->getFirstName()): null,
                'lastName' => $epakaSender->getLastName() ? ucfirst($epakaSender->getLastName()): null,
                'street' => $epakaSender->getStreet() ?: null,
                'houseNumber' => $epakaSender->getHouseNumber() ?: null,
                'postCode' => $epakaSender->getPostalCode() ?: null,
                'city' => $epakaSender->getCity() ?: null,
                'country' => Country::getNameById($psOrder->id_lang, $countryId),
                'phone' =>  $epakaSender->getPhone() ?: null,
                'email' => $epakaSender->getEmail() ?: null,
            ],
            'orderAdditionalOptions' => $orderAdditionalOptions,
            'carrierAdditionalMessage' => $psOrder->getFirstMessage() ?: 'Brak informacji',
            'totalPaidTaxIncl' => round($psOrder->total_paid_tax_incl, 2) . 'zł',
            'isHasInvoice' => $this->getIsOrderHasInvoice($psOrder),
            'invoiceData' => $invoiceData,
            'orderCoreEditUrl' => $this->getLinkToOrderPsCorePage($this->context, (int) $psOrder->id),
            'deleteOrderUrl' => $this->generateDeleteOrderUrl((int) $psOrder->id),
            'deleteShipmentUrl' => $epakaParcel ? $this->generateDeleteApiShipmentUrl((int) $epakaParcel->getApiOrderId()) : null,
            'pointTrackingCode' => $trackingNumber,
            'templateName' => $psCarrier->name,
            'shipmentType' => ShippingMapper::convertShippingTypeToHumanString($epakaCarrier->getShipmentType()),
            'totalLength' => $totalLength,
            'totalWidth' => $totalWidth,
            'totalHeight' => $totalHeight,
            'totalOrderWeight' => $this->getTotalOrderWeight($psOrder),
            'isShipmentPage' => (bool) $isShipmentPage,
            'epakaOrderId' => $epakaParcel ? $epakaParcel->getApiOrderId() : null,
            'downloadProtocolUrl' => $epakaParcel && $epakaParcel->getApiOrderId() ? $this->getOrderDownloadUrl($epakaParcel->getApiOrderId(), 'protocol') : null,
            'downloadLabelUrl' => $epakaParcel && $epakaParcel->getApiOrderId() ? $this->getOrderDownloadUrl($epakaParcel->getApiOrderId(), DocumentsLabelFormatMapper::convertIntToStringStatus($epakaSettings->getDefaultLabelFormat())) : null
        ];
        return new Response(json_encode($additionalInfo));
    }

    public function documentDownloadAction(Request $request): Response
    {
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        $token = $request->get('_token');
        $id = $request->query->get('id');
        $type = $request->query->get('type');
        try {
            $orderDocumentInBase64 = ApiService::fetchOrderDocument($authGuardService->getAccessToken(), $id, $type);
            if(is_array($orderDocumentInBase64) && isset($orderDocumentInBase64['document'])) {
                $decodedPdf = base64_decode($orderDocumentInBase64['document']);
                $response = new Response($decodedPdf);
                $response->headers->set('Content-Type', 'application/pdf');
                $response->headers->set('Content-Disposition', 'attachment; filename="' . $type . '.pdf"');
                $response->headers->set('Content-Length', strlen($decodedPdf));
                $response->headers->set('Cache-Control', 'private');
                $response->headers->set('Pragma', 'no-cache');
            } else {
                $response = new Response();
            }
            return $response;
        } catch (Exception $e) {
            $errorRaw = json_decode($e->getMessage(), true);
            $error = LoggerService::normalizeErrorToFlashMessage($errorRaw, 'Błąd pobierania dokumentu');
            LoggerService::logError(
                $error['message'],
                $e->getCode() ?? 0,
                $e->getFile(),
                $e->getLine()
            );
            $this->addFlash('error', $error['message']);
            return $this->redirectToRoute('list_of_shipment');
        }
    }

    public function apiCreateAction(Request $request): Response
    {
        /**
         * @var $epakaOrder EpakaOrder
         * @var $epakaCarrier EpakaCarrier
         * @var $epakaSender EpakaSender
         * @var $epakaSettings EpakaSettings
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            $this->addFlash('warning', 'Sesja wygasła');
            return new JsonResponse(
                [
                    'redirectUrl' => $authGuardService->getPageUrl(),
                ], 401
            );
        }
        $em = $this->getDoctrine()->getManager();
        $auth = $em->getRepository(EpakaAuth::class)->findOneBy([]) ?? null;
        $data = json_decode($request->getContent(), true);
        $epakaOderId = $data['epakaOderId'] ?? null;
        $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaOderId]);
        $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
        if(!$epakaOrder) {
            LoggerService::logError(
                'EpakaOrder instance not found with ID: ' . $epakaOderId,
                500,
                __FILE__,
                "(EpakaOrder instance not found while creating an order on Epaka)"
            );
            return new JsonResponse([
                'message' => 'Wystąpił błąd podczas tworzenia zamówienia na platformie Epaka',
            ], 404);
        }
        try {
            $userInfoData = ApiService::fetchUserInfo($authGuardService->getAccessToken());
        } catch (Exception $e) {
            $errorRaw = json_decode($e->getMessage(), true);
            $error = LoggerService::normalizeErrorToFlashMessage($errorRaw, 'Błąd tworzenia zamówienia');
            LoggerService::logError(
                $error['message'],
                $e->getCode() ?? 0,
                $e->getFile(),
                $e->getLine()
            );
            return new JsonResponse([
                'message' => $error['message'],
            ], 502);
        }
        // If you want to validate an entire array (e.g., all fields in "senderData" or "invoiceData"),
        // add it to $requiredGroups.
        // Example:
        // $requiredGroups = ['senderData', 'invoiceData']; // This will check all fields in these arrays.
        $requiredGroups = [];
        // If you only need to validate specific fields, add them to $requiredFields using dot notation.
        // Example:
        // $requiredFields = ['senderData.name', 'invoiceData.email', 'nip'];
        // This will check only these specific fields.
        $requiredFields = [
            'email',
            'senderData.name',
            'senderData.lastName',
            'senderData.country',
            'senderData.postCode',
            'senderData.city',
            'senderData.street',
            'senderData.houseNumber',
            'senderData.phone',
            'invoiceData.name',
            'invoiceData.lastName',
            'invoiceData.country',
            'invoiceData.postCode',
            'invoiceData.city',
            'invoiceData.street',
            'invoiceData.houseNumber',
            'invoiceData.phone',
        ];
        $isAllRequiredUserInfoFieldsFilled = $this->validateRequiredUserInfoFields($userInfoData, $requiredGroups, $requiredFields);
        if(!$isAllRequiredUserInfoFieldsFilled) {
            return new JsonResponse([
                'message' => 'Niektóre wymagane dane użytkownika na Epaka.pl nie zostały uzupełnione. Zaloguj się do swojego konta na Epaka.pl, uzupełnij brakujące informacje i spróbuj ponownie.',
            ], 409);
        }
        $psOrder = new Order($epakaOrder->getPsOrderId());
        $trackingNumber = Db::getInstance()->getValue('
            SELECT tracking_number
            FROM `' . _DB_PREFIX_ . 'order_carrier`
            WHERE id_order = ' . (int)$psOrder->id
        );
        if (!$trackingNumber) $trackingNumber = null;
        $psCarrier = new Carrier($psOrder->id_carrier);
        $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['psCarrierId' => $psCarrier->id]);
        $epakaSender = $em->getRepository(EpakaSender::class)->findOneBy(['carrierId' => $epakaCarrier->getId()]);
        $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['psOrderId' => $psOrder->id]);
        $cart = new Cart($psOrder->id_cart);
        $deliveryAddress = new Address($cart->id_address_delivery);
        $customer = new Customer($deliveryAddress->id_customer);
        $idOrder = (int)$psOrder->id;
//        $sql = 'SELECT `message`
//                FROM `' . _DB_PREFIX_ . 'message`
//                WHERE `id_order` = ' . $idOrder . '
//                  AND `id_employee` = 0
//                ORDER BY `date_add` DESC';
//        $carrierComment = Db::getInstance()->getValue($sql);
        $carrierComment = $psOrder->getFirstMessage();
        try {
            $totalOrderWeight = $this->getTotalOrderWeight($psOrder);
            $dhlType = $this->getDhlType($epakaCarrier->getEpakaCurierId(), $epakaCarrier->getShipmentType(), $epakaCarrier->getEpakaCourierName(), $totalOrderWeight);
            [$upsServiceType, $upsPackagesNumber, $upsWeight] = $this->getUpsData($epakaCarrier, $totalOrderWeight);
            $insuranceAmount = $this->getInsuranceAmount($epakaCarrier->getAdditionalOptions());
            $additionalOptionsNameCollection = $this->getOrderAdditionalOptionsApiNameCollection($epakaCarrier->getAdditionalOptions(), ['insurance', 'cod']);
            $codeAmount = $this->getCodAmount($epakaCarrier->getAdditionalOptions());
            $paymentType = PaymentTypeMapper::convertIntToStringStatus($epakaSettings->getPaymentType());
            $response = ApiOrderService::create(
                $auth->getAccessToken(),
                $epakaCarrier->getShipmentType(),
                $epakaCarrier->getParcelType(),
                $paymentType,
                $psOrder->id ? sprintf('Numer zamówienia w sklepie: %s', $psOrder->id) : 'Brak numeru zamówienia w sklepie',
                $cart->getProducts(),
                $epakaCarrier->getEpakaCurierId(),
                (float)$psCarrier->max_height,
                (float)$psCarrier->max_depth,
                (float)$psCarrier->max_width,
                (float)$cart->getTotalWeight(),
                substr(trim($deliveryAddress->city), 0, ApiFieldLengthService::receiverCityMax()),
                (new Country($deliveryAddress->id_country))->iso_code,
                trim($customer->email),
                substr(trim($deliveryAddress->address2), 0, ApiFieldLengthService::receiverAddress2Max()) ?: substr(trim($deliveryAddress->address1), 0, ApiFieldLengthService::receiverAddress2Max()),
                substr(trim($deliveryAddress->firstname), 0, ApiFieldLengthService::receiverFirstnameMax()),
                substr(trim($deliveryAddress->lastname), 0, ApiFieldLengthService::receiverLastnameMax()),
                substr(trim($deliveryAddress->phone), 0, ApiFieldLengthService::receiverPhoneMax()),
                substr(trim($deliveryAddress->postcode), 0, ApiFieldLengthService::receiverPostcodeMax()),
                substr($deliveryAddress->address1, 0, ApiFieldLengthService::receiverAddress1Max()),
                trim($deliveryAddress->vat_number) ?: null,
                substr(trim($deliveryAddress->company), 0, ApiFieldLengthService::receiverCompanyMax()) ?: null,
                $epakaSender,
                $carrierComment,
                $insuranceAmount,
                $codeAmount,
                $additionalOptionsNameCollection,
                $codeAmount ? "account" : null,
                $trackingNumber,
                $dhlType,
                $upsServiceType,
                $upsPackagesNumber,
                $upsWeight
            );
            $message = $response['message'] ?? null;
            $apiOrderId = $response['orderId'] ?? null;
            $epakaOrderDetails = ApiService::fetchOrderDetails($authGuardService->getAccessToken(), $apiOrderId);
            if($epakaOrderDetails) {
                $epakaParcel = new EpakaParcel();
                $epakaParcel->setApiOrderId($apiOrderId);
                $epakaParcel->setCourierStatus(isset($epakaOrderDetails['packageStatus']) ? CourierStatusMapper::convertStringStatusToInt($epakaOrderDetails['packageStatus']) : CourierStatusMapper::PACKAGE_NO_SEND);
                $epakaParcel->setEpakaOrderId($epakaOrder->getId());
                $epakaParcel->setEpakaPlacedAt(new DateTime($epakaOrderDetails['orderDate']));
                $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::convertStringStatusToInt($epakaOrderDetails['orderStatus']));
                $epakaParcel->setMessage($message);
                $epakaParcel->setOrderProcessResultStatus($response['orderProcessResult']);
                $epakaParcel->setWaybillNumber($epakaOrderDetails['labelNumber'] ?? null);
                $em->persist($epakaParcel);
                $em->flush();
            }
            $em->flush();
            return new JsonResponse([
                'message' => $response['message'],
            ], 200);

        } catch (Exception $e) {
            $errorRaw = json_decode($e->getMessage(), true);
            $error = LoggerService::normalizeErrorToFlashMessage($errorRaw, 'Błąd tworzenia zamówienia');
            LoggerService::logError(
                $error['message'],
                $e->getCode() ?? 0,
                $e->getFile(),
                $e->getLine()
            );
            return new JsonResponse([
                'message' => $error['message'],
            ], ($e->getCode() > 0) ? $e->getCode() : 400);
        }
    }
    function validateRequiredUserInfoFields(array $data, array $requiredGroups = [], array $requiredFields = []): bool
    {
        foreach ($requiredGroups as $group) {
            if (!isset($data[$group]) || !is_array($data[$group])) {
                return false;
            }
            foreach ($data[$group] as $key => $value) {
                if ($value === null) {
                    return false;
                }
            }
        }
        foreach ($requiredFields as $field) {
            $keys = explode('.', $field);
            $value = $data;

            foreach ($keys as $key) {
                if (!isset($value[$key])) {
                    return false;
                }
                $value = $value[$key];
            }

            if ($value === null) {
                return false;
            }
        }

        return true;
    }

    public function deleteOrderAction(Request $request): Response
    {
        /**
         * @var $epakaOrder EpakaOrder
         */
        $id = $request->get('id');
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        $em = $this->getDoctrine()->getManager();
        $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['psOrderId' => (int)$id]);
        $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['epakaOrderId' => $epakaOrder->getId(), 'isDeleted' => false]);
        if (!$epakaOrder) {
            $this->addFlash('error', 'Zamówienie nie zostało znalezione');
            return $this->redirectToRoute('list_of_order');
        }
        $order = new Order($id);
        if (!Validate::isLoadedObject($order)) {
            $this->addFlash('error', 'Zamówienie nie zostało znalezione w PrestaShop');
            return $this->redirectToRoute('list_of_order');
        }
        $canceledStatus = (int) Configuration::get('PS_OS_CANCELED');
        if (!$canceledStatus) {
            $this->addFlash('error', 'Nie znaleziono statusu anulowania zamówienia');
            return $this->redirectToRoute('list_of_order');
        }
        $history = new OrderHistory();
        $history->id_order = (int) $order->id;
        $history->changeIdOrderState($canceledStatus, (int) $order->id);
        $history->addWithemail(true);
        if ($epakaOrder) {
            $epakaOrder->setIsDeleted(true);
            $em->flush();
        }
        try {
            if($epakaParcel) {
                $response = ApiService::cancelOrder($authGuardService->getAccessToken(), $epakaParcel->getApiOrderId());
                if (!empty($response['message'])) {
                    $this->addFlash('success', $response['message']);
                } else {
                    $this->addFlash('success', 'Zamówienie zostało anulowane.');
                }
                $epakaParcel->setIsDeleted(true);
                $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
                $epakaOrder->setUpdatedAt(new DateTime('now'));
                $em->flush();
            }
        } catch (Exception $e) {
            if ($e->getCode() === 409) {
                $this->addFlash('warning', 'Zamówienie już zostało anulowane.');
                $epakaOrder->setIsDeleted(true);
                $epakaOrder->setUpdatedAt(new DateTime('now'));
                $em->flush();
            } else {
                LoggerService::logError(
                    $e->getMessage(),
                    $e->getCode(),
                    $e->getFile(),
                    $e->getLine()
                );
                $this->addFlash('error', 'Błąd: ' . $e->getMessage());
            }
        }

        return $this->redirectToRoute('list_of_order');
    }

    public function deleteShipmentAction(Request $request): Response
    {
        /**
         * @var $epakaOrder EpakaOrder
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) {
            return $this->redirectToRoute('auth');
        }
        $epakaOrderId = $request->get('id');
        $em = $this->getDoctrine()->getManager();
        $epakaParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['apiOrderId' => (int) $epakaOrderId]);
        if (!$epakaParcel) {
            $this->addFlash('error', 'Zamówienie nie zostało znalezione');
            return $this->redirectToRoute('list_of_order');
        }
        try {
            $response = ApiService::cancelOrder($authGuardService->getAccessToken(), $epakaOrderId);
            $epakaParcel->setIsDeleted(true);
            $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
            $em->flush();
            if (!empty($response['message'])) {
                $this->addFlash('success', $response['message']);
            } else {
                $this->addFlash('success', 'Zamówienie zostało anulowane.');
            }
        } catch (Exception $e) {
            if ($e->getCode() === 409) {
                $this->addFlash('warning', 'Zamówienie już zostało anulowane.');
                $epakaParcel->setIsDeleted(true);
                $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::CANCELED);
                $em->flush();
            } else {
                LoggerService::logError(
                    $e->getMessage(),
                    $e->getCode(),
                    $e->getFile(),
                    $e->getLine()
                );
                $this->addFlash('error', 'Błąd: ' . $e->getMessage());
            }
        }

        return $this->redirectToRoute('list_of_shipment');
    }

    public function massAction(Request $request): Response
    {
        /**
         * @var $epakaOrder EpakaOrder
         * @var $epakaParcel EpakaParcel
         */
        $authGuardService = $this->get('epaka.service.auth_guard');
        if (!$authGuardService->isAuthenticated()) return $this->redirectToRoute('auth');
        $massEpakaGenerateType = 'create-epaka-orders';
        $massPsAndEpakaCancelType = 'delete-ps-and-epaka-orders';
        $massType = $request->request->get('massType');
        $recordsIds = json_decode($request->request->get('recordsIds'), true) ?? [];
        $em = $this->getDoctrine()->getManager();
        $rowErrors = [];
        $errorFlashMessage = null;
        $errorFlashMessageType= null;
        if($massType === $massEpakaGenerateType) {
            foreach ($recordsIds as $recordsId) {
                $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $recordsId]);
                $psOrder = new Order($epakaOrder->getPsOrderId());
                if($epakaOrder  && !$this->isOrderCanceled($psOrder)) {
                    $epakaOderId = $recordsId;
                    $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => $epakaOderId]);
                    $epakaOrderParcel = $em->getRepository(EpakaParcel::class)->findOneBy(['epakaOrderId' => $epakaOrder->getId(), 'isDeleted' => false]);
                    $epakaSettings = $em->getRepository(EpakaSettings::class)->findOneBy([]) ?? null;
                    if(!$epakaOrder) {
                        LoggerService::logError(
                            'EpakaOrder instance not found with ID: ' . $epakaOderId,
                            500,
                            __FILE__,
                            "(EpakaOrder instance not found while creating an order on Epaka)"
                        );
                        $rowErrors[] = ['id' => $recordsId,' message' => 'EpakaOrder instance not found'];
                    } elseif ($epakaOrderParcel) {
                        continue;
                    }
                    else {
                        try {
                            $userInfoData = ApiService::fetchUserInfo($authGuardService->getAccessToken());
                        } catch (Exception $e) {
                            $errorRaw = json_decode($e->getMessage(), true);
                            $error = LoggerService::normalizeErrorToCollection((int)$recordsId, $errorRaw, 'Nie udało się pobrać informacji o użytkowniku');
                            LoggerService::logError(
                                $error['message'],
                                $e->getCode() ?? 0,
                                $e->getFile(),
                                $e->getLine()
                            );
                            LoggerService::logError(
                                $error['message'],
                                $e->getCode() ?? 0,
                                $e->getFile(),
                                $e->getLine()
                            );
                            $rowErrors[] = ['id' => $error['id'],' message' => $error['message']];
                            continue;
                        }
                        // If you want to validate an entire array (e.g., all fields in "senderData" or "invoiceData"),
                        // add it to $requiredGroups.
                        // Example:
                        // $requiredGroups = ['senderData', 'invoiceData']; // This will check all fields in these arrays.
                        $requiredGroups = [];
                        // If you only need to validate specific fields, add them to $requiredFields using dot notation.
                        // Example:
                        // $requiredFields = ['senderData.name', 'invoiceData.email', 'nip'];
                        // This will check only these specific fields.
                        $requiredFields = [
                            'email',
                            'senderData.name',
                            'senderData.lastName',
                            'senderData.country',
                            'senderData.postCode',
                            'senderData.city',
                            'senderData.street',
                            'senderData.houseNumber',
                            'senderData.phone',
                            'invoiceData.name',
                            'invoiceData.lastName',
                            'invoiceData.country',
                            'invoiceData.postCode',
                            'invoiceData.city',
                            'invoiceData.street',
                            'invoiceData.houseNumber',
                            'invoiceData.phone',
                        ];
                        $isAllRequiredUserInfoFieldsFilled = $this->validateRequiredUserInfoFields($userInfoData, $requiredGroups, $requiredFields);
                        if(!$isAllRequiredUserInfoFieldsFilled) {
                            $rowErrors[] = ['id' => $recordsId,' message' => 'Nie wszystkie obowiązkowe pola na koncie użytkownika epaka są wypełnione'];
                            continue;
                        }
                        $trackingNumber = Db::getInstance()->getValue('SELECT tracking_number FROM `' . _DB_PREFIX_ . 'order_carrier` WHERE id_order = ' . (int)$psOrder->id);
                        if (!$trackingNumber) $trackingNumber = null;
                        $psCarrier = new Carrier($psOrder->id_carrier);
                        $epakaCarrier = $em->getRepository(EpakaCarrier::class)->findOneBy(['psCarrierId' => $psCarrier->id]);
                        $epakaSender = $em->getRepository(EpakaSender::class)->findOneBy(['carrierId' => $epakaCarrier->getId()]);
                        $cart = new Cart($psOrder->id_cart);
                        $deliveryAddress = new Address($cart->id_address_delivery);
                        $customer = new Customer($deliveryAddress->id_customer);
                        $carrierComment = $psOrder->getFirstMessage();
                        try {
                            $totalOrderWeight = $this->getTotalOrderWeight($psOrder);
                            $dhlType = $this->getDhlType($epakaCarrier->getEpakaCurierId(), $epakaCarrier->getShipmentType(), $epakaCarrier->getEpakaCourierName(), $totalOrderWeight);
                            [$upsServiceType, $upsPackagesNumber, $upsWeight] = $this->getUpsData($epakaCarrier, $totalOrderWeight);
                            $insuranceAmount = $this->getInsuranceAmount($epakaCarrier->getAdditionalOptions());
                            $additionalOptionsNameCollection = $this->getOrderAdditionalOptionsApiNameCollection($epakaCarrier->getAdditionalOptions(), ['insurance', 'cod']);
                            $codeAmount = $this->getCodAmount($epakaCarrier->getAdditionalOptions());
                            $paymentType = PaymentTypeMapper::convertIntToStringStatus($epakaSettings->getPaymentType());
                            $response = ApiOrderService::create(
                                $authGuardService->getAccessToken(),
                                $epakaCarrier->getShipmentType(),
                                $epakaCarrier->getParcelType(),
                                $paymentType,
                                $psOrder->id ? sprintf('Numer zamówienia w sklepie: %s', $psOrder->id) : 'Brak numeru zamówienia w sklepie',
                                $cart->getProducts(),
                                $epakaCarrier->getEpakaCurierId(),
                                (float)$psCarrier->max_height,
                                (float)$psCarrier->max_depth,
                                (float)$psCarrier->max_width,
                                (float)$cart->getTotalWeight(),
                                substr(trim($deliveryAddress->city), 0, ApiFieldLengthService::receiverCityMax()),
                                (new Country($deliveryAddress->id_country))->iso_code,
                                trim($customer->email),
                                substr(trim($deliveryAddress->address2), 0, ApiFieldLengthService::receiverAddress2Max()) ?: substr(trim($deliveryAddress->address1), 0, ApiFieldLengthService::receiverAddress2Max()),
                                substr(trim($deliveryAddress->firstname), 0, ApiFieldLengthService::receiverFirstnameMax()),
                                substr(trim($deliveryAddress->lastname), 0, ApiFieldLengthService::receiverLastnameMax()),
                                substr(trim($deliveryAddress->phone), 0, ApiFieldLengthService::receiverPhoneMax()),
                                substr(trim($deliveryAddress->postcode), 0, ApiFieldLengthService::receiverPostcodeMax()),
                                substr($deliveryAddress->address1, 0, ApiFieldLengthService::receiverAddress1Max()),
                                trim($deliveryAddress->vat_number) ?: null,
                                substr(trim($deliveryAddress->company), 0, ApiFieldLengthService::receiverCompanyMax()) ?: null,
                                $epakaSender,
                                $carrierComment,
                                $insuranceAmount,
                                $codeAmount,
                                $additionalOptionsNameCollection,
                                $codeAmount ? "account" : null,
                                $trackingNumber,
                                $dhlType,
                                $upsServiceType,
                                $upsPackagesNumber,
                                $upsWeight
                            );
                            $message = $response['message'] ?? null;
                            $apiOrderId = $response['orderId'] ?? null;
                            $epakaOrderDetails = ApiService::fetchOrderDetails($authGuardService->getAccessToken(), $apiOrderId);
                            if($epakaOrderDetails) {
                                $epakaParcel = new EpakaParcel();
                                $epakaParcel->setApiOrderId($apiOrderId);
                                $epakaParcel->setCourierStatus(isset($epakaOrderDetails['packageStatus']) ? CourierStatusMapper::convertStringStatusToInt($epakaOrderDetails['packageStatus']) : CourierStatusMapper::PACKAGE_NO_SEND);
                                $epakaParcel->setEpakaOrderId($epakaOrder->getId());
                                $epakaParcel->setEpakaPlacedAt(new DateTime($epakaOrderDetails['orderDate']));
                                $epakaParcel->setEpakaStatus(EpakaOrderStatusMapper::convertStringStatusToInt($epakaOrderDetails['orderStatus']));
                                $epakaParcel->setMessage($message);
                                $epakaParcel->setOrderProcessResultStatus($response['orderProcessResult']);
                                $epakaParcel->setWaybillNumber($epakaOrderDetails['labelNumber'] ?? null);
                                $em->persist($epakaParcel);
                                $em->flush();
                            }
                            $em->flush();
                            continue;
                        } catch (Exception $e) {
                            $errorsRaw = json_decode($e->getMessage(), true);
                            LoggerService::logError(
                                $e->getMessage(),
                                $e->getCode() ?? 0,
                                $e->getFile(),
                                $e->getLine()
                            );
                            $error = LoggerService::normalizeErrorToCollection((int)$recordsId, $errorsRaw, 'Błąd tworzenia zamówienia');
                            $rowErrors[] = ['id' => $error['id'], 'message' => $error['message']];
                        }
                    }
                }
            }
            $flashError = LoggerService::normalizeCollectionFlashMessages($rowErrors, 'Wystąpiły błędy podczas tworzenia zamówień', 'Wszystkie zamówienia zostały pomyślnie utworzone', 'error');
            if($flashError) {
                $errorFlashMessage = $flashError['message'];
                $errorFlashMessageType = $flashError['type'];
            }
        } elseif ($massType === $massPsAndEpakaCancelType) {
            foreach ($recordsIds as $recordsId) {
                $epakaOrder = $em->getRepository(EpakaOrder::class)->findOneBy(['id' => (int)$recordsId]);
                if (!$epakaOrder) {
                    LoggerService::logError(
                        'Zamówienie nie zostało znalezione',
                        404,
                        '',
                        ''
                    );
                    $rowErrors[] = ['id' => $recordsId,' message' => 'Zamówienie nie zostało znalezione'];
                    continue;
                }
                $epakaOrderParcels = $em->getRepository(EpakaParcel::class)->findBy(['epakaOrderId' => $recordsId, 'isDeleted' => false]);
                foreach ($epakaOrderParcels as $epakaParcel) {
                    if(!$epakaParcel->isDeleted()) {
                        try {
                            ApiService::cancelOrder($authGuardService->getAccessToken(), $epakaParcel->getApiOrderId());
                            $epakaOrder->setIsDeleted(true);
                            $em->flush();
                        } catch (Exception $e) {
                            if ($e->getCode() === 409) {
                                $this->addFlash('warning', 'Zamówienie już zostało anulowane.');
                                $epakaOrder->setIsDeleted(true);
                                continue;
                            } else {
                                LoggerService::logError(
                                    $e->getMessage(),
                                    $e->getCode(),
                                    $e->getFile(),
                                    $e->getLine()
                                );
                                $errorsRaw = json_decode($e->getMessage(), true);
                                $error = LoggerService::normalizeErrorToCollection((int)$recordsId, $errorsRaw, 'Błąd usuwania zamówienia');
                                $rowErrors[] = ['id' => $error['id'], 'message' => $error['message']];
                                continue;
                            }
                        }
                    }
                }
                $psOrder = new Order($epakaOrder->getPsOrderId());
                if($this->isOrderCanceled($psOrder)) continue;
                if (!Validate::isLoadedObject($psOrder)) {
                    LoggerService::logError(
                        'Zamówienie nie zostało znalezione w PrestaShop',
                        404,
                        '',
                        ''
                    );
                    $rowErrors[] = ['id' => $recordsId,' message' => 'Zamówienie nie zostało znalezione w PrestaShop'];
                    continue;
                }
                $canceledStatus = (int) Configuration::get('PS_OS_CANCELED');
                if (!$canceledStatus) {
                    LoggerService::logError(
                        'Nie znaleziono statusu anulowania zamówienia',
                        404,
                        '',
                        ''
                    );
                    $rowErrors[] = ['id' => $recordsId,' message' => 'Nie znaleziono statusu anulowania zamówienia'];
                    continue;
                }
                $history = new OrderHistory();
                $history->id_order = (int) $psOrder->id;
                $history->changeIdOrderState($canceledStatus, (int) $psOrder->id);
                $history->addWithemail(true);
                $epakaOrder->setIsDeleted(true);
                $em->flush();
            }
            $flashError = LoggerService::normalizeCollectionFlashMessages($rowErrors, 'Wystąpiły błędy podczas usuwania zamówień', 'Wszystkie zamówienia zostały pomyślnie usunięte', 'error');
            if($flashError) {
                $errorFlashMessage = $flashError['message'];
                $errorFlashMessageType = $flashError['type'];
            }
        }

        if ($errorFlashMessageType && $errorFlashMessage) {
            $this->addFlash($errorFlashMessageType, $errorFlashMessage);
        }

        return $this->redirectToRoute('list_of_order');
    }

    public function getTotalOrderWeight($psOrder): float
    {
        $orderProducts = $psOrder->getProducts();
        $totalOrderWeight = 0;
        foreach ($orderProducts as $product) {
            $totalOrderWeight += $product['product_weight'] * $product['product_quantity'];
        }
        return $totalOrderWeight;
    }
    public function getOrderPackageDimensions($psOrder): array
    {
        $totalLength = 0;
        $totalWidth  = 0;
        $totalHeight = 0;
        $orderProducts = $psOrder->getProducts();
        foreach ($orderProducts as $orderProduct) {
            $product = new Product($orderProduct['product_id'], false, $psOrder->id_lang);
            $totalLength += $product->width;
            $totalWidth = max($totalWidth, $product->height);
            $totalHeight = max($totalHeight, $product->depth);
        }
        return [round($totalLength, 2), round($totalWidth, 2), round($totalHeight, 2)];
    }
    private function getDhlType(int $epakaCourierId, int $shippingType, string $epakaCourierName, float $totalOrderWeight): ?string
    {
        $dhlType = null;
        if($epakaCourierId === self::DHL_API_ID || $epakaCourierId === self::DHL_SERVICE_POINT_API_ID) {
            switch (true) {
                case ShippingMapper::convertShippingTypeToString($shippingType) === ShippingMapper::PACKAGE_TYPE_PALLET && $totalOrderWeight > self::DHL_TYPE_WEIGHT:
                case ShippingMapper::convertShippingTypeToString($shippingType) === ShippingMapper::PACKAGE_TYPE_PARCEL && $totalOrderWeight > self::DHL_TYPE_WEIGHT: {
                    $dhlType = "dr";
                    break;
                }
                case ShippingMapper::convertShippingTypeToString($shippingType) === ShippingMapper::PACKAGE_TYPE_PARCEL && $totalOrderWeight <= self::DHL_TYPE_WEIGHT:
                case ShippingMapper::convertShippingTypeToString($shippingType) === ShippingMapper::PACKAGE_TYPE_ENVELOPE && $totalOrderWeight <= self::DHL_TYPE_WEIGHT: {
                    $dhlType = "ex";
                    break;
                }
                case $epakaCourierName === "DHL Express": {
                    $dhlType = "express";
                    break;
                }
            }
        }
        return $dhlType;
    }
    public function getUpsData($epakaCarrier, $totalOrderWeight): array
    {
        $upsServiceType   = null;
        $upsPackagesNumber = null;
        $upsWeight        = null;
        switch (true) {
            case $epakaCarrier->getEpakaCourierName() === self::UPS_STANDARD:
                $upsServiceType = self::UPS_STANDARD;
                break;
            case $epakaCarrier->getEpakaCourierName() === self::UPS_EXPRESS_SAVER:
                $upsServiceType = self::UPS_EXPRESS_SAVER;
                break;
            case $epakaCarrier->getEpakaCourierName() === self::UPS_EXPRESS_PLUS:
                $upsServiceType = self::UPS_EXPRESS_PLUS;
                break;
            case $epakaCarrier->getEpakaCourierName() === self::UPS_EXPRESS:
                $upsServiceType = self::UPS_EXPRESS;
                break;
        }
        if (stripos($epakaCarrier->getEpakaCourierName(), 'UPS') !== false) {
            $upsPackagesNumber = 1;
            $upsWeight = $totalOrderWeight;
        }

        return [$upsServiceType, $upsPackagesNumber, $upsWeight];
    }
    private function getAdminUrls(): array
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            $adminUrl = rtrim(Tools::getHttpHost(true), '/');
            return [
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaOrderAdditionalInfo', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaOrderApiCreate', true),
                $adminUrl . $this->context->link->getAdminLink('AdminEpakaOrderApiMass', true),
            ];
        } else {
            return [
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderAdditionalInfo', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaCouriers'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderApiCreate', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaOrderApiCreate'),
                '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderApiMass', true), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaOrderApiMass'),
            ];
        }
    }
    private function getOrderDownloadUrl(int $id, string $type): string
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            $url = rtrim(Tools::getHttpHost(true), '/') . '/' .
                ltrim($this->context->link->getAdminLink('AdminEpakaOrderDownload', true), '/');
            $url = str_replace('&amp;', '&', $url);
            return $url . '&id=' . (int)$id . '&type=' . urlencode($type);
        } else {
            $baseUrl = '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaOrderDownload', true), PHP_URL_PATH), '/');
            $token = Tools::getAdminTokenLite('AdminEpakaOrderDownload');

            return $baseUrl . '?_token=' . $token . '&id=' . (int)$id . '&type=' . urlencode($type);
        }
    }
    private function getLinkToOrderPsCorePage(Context $context, int $psOrderId): string
    {
        $adminDir = basename(_PS_ADMIN_DIR_);
        $adminBaseUrl = $context->shop->getBaseURL(true);
        $psVersion = _PS_VERSION_;

        if (version_compare($psVersion, '1.7.8', '>=')) {
            $token = Tools::getAdminTokenLite('AdminOrders');
            $relativeAdminOrderUrl = 'index.php?controller=AdminOrders&id_order=' . $psOrderId . '&vieworder&token=' . $token;
            return $adminBaseUrl .  $adminDir . '/' . ltrim($relativeAdminOrderUrl, '/');
        } else {
            // PrestaShop 1.7.6 - 1.7.7
            $token = Tools::getAdminTokenLite('AdminOrders');
            $relativeAdminOrderUrl = 'index.php?controller=AdminOrders&id_order=' . $psOrderId . '&vieworder&token=' . $token;
            return $adminBaseUrl .  $adminDir . '/' . ltrim($relativeAdminOrderUrl, '/');
        }

    }
    private function generateDeleteOrderUrl(int $psOrderId): string
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            return rtrim(Tools::getHttpHost(true), '/') . '/' . ltrim($this->context->link->getAdminLink('AdminEpakaDeleteOrder', true, ['id' => $psOrderId]), '/');
        } else {
            return '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaDeleteOrder', true, ['id' => $psOrderId]), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaDeleteOrder');
        }
    }
    private function generateDeleteApiShipmentUrl(int $apiOrderId): string
    {
        if (version_compare(_PS_VERSION_, '1.7.6.8', '<') || version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            return rtrim(Tools::getHttpHost(true), '/') . '/' . ltrim($this->context->link->getAdminLink('AdminEpakaDeleteShipment', true, ['id' => $apiOrderId]), '/');
        } else {
            return '/' . ltrim(parse_url($this->context->link->getAdminLink('AdminEpakaDeleteShipment', true, ['id' => $apiOrderId]), PHP_URL_PATH), '/') . '?_token=' . Tools::getAdminTokenLite('AdminEpakaDeleteShipment');
        }
    }
    public function getAllOrderStatuses(): array
    {
        $orderStates = OrderState::getOrderStates((int)$this->context->language->id);
        $statuses = [];
        foreach ($orderStates as $state) {
            $statuses[$state['id_order_state']] = $state['name'];
        }

        return $statuses;
    }
    private function getCodAmount(?array $additionalOptions): ?float
    {
        if (!empty($additionalOptions) && isset($additionalOptions['cod'])) {
            return (float)$additionalOptions['cod']['amount'] ?? null;
        }
        return null;
    }
    private function getInsuranceAmount(?array $additionalOptions): ?float
    {
        if (!empty($additionalOptions) && isset($additionalOptions['insurance'])) {
            return (float)$additionalOptions['insurance']['amount'] ?? null;
        }
        return null;
    }
    private function getOrderAdditionalOptionsApiNameCollection(?array $additionalOptions, array $excluded = []): ?array
    {
        if (!empty($additionalOptions) && is_array($additionalOptions)) {
            $additionalOptionsArray = [];
            foreach ($additionalOptions as $key => $value) {
                if(!empty($excluded) && in_array($key, $excluded, true)) continue;
                $additionalOptionsArray[] = $key;
            }
            return $additionalOptionsArray;
        }
        return null;
    }
    private function isOrderCanceled($psOrder): bool
    {
        $currentStateId = $psOrder->getCurrentState();
        $orderState = new OrderState($currentStateId, $this->context->language->id);
        $cancelState = new OrderState(Configuration::get('PS_OS_CANCELED'), (int)$this->context->language->id);
        return $orderState->name === $cancelState->name;
    }
    private function getIsOrderHasInvoice($order): bool
    {
        $isHasInvoice = false;
        $idAddress = $order->id_address_delivery;
        $address = new Address((int)$idAddress);
        if($address->vat_number) {
            $isHasInvoice = true;
        } else {
            $idAddress = $order->id_address_invoice;
            $address = new Address((int)$idAddress);
            $isHasInvoice = (int)$address->vat_number > 0;
        }
        return $isHasInvoice;
    }
}